<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

if (!$usuario_id) {
    $response['message'] = 'Acceso denegado.';
    echo json_encode($response);
    exit();
}

try {
    switch ($action) {
        case 'cargar_todos_los_usuarios':
            $stmt = $conn->prepare("SELECT id, nombre, foto_perfil FROM usuarios WHERE estado = 1 AND id != ? ORDER BY nombre ASC");
            $stmt->execute([$usuario_id]);
            $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $usuarios];
            break;

        case 'cargar_conversaciones':
            $stmt = $conn->prepare("
                SELECT
                    m.id, m.asunto, m.tipo,
                    (SELECT mm.mensaje FROM memorandos_mensajes mm WHERE mm.memorando_id = m.id ORDER BY mm.fecha_envio DESC LIMIT 1) as ultimo_mensaje,
                    (SELECT mm.fecha_envio FROM memorandos_mensajes mm WHERE mm.memorando_id = m.id ORDER BY mm.fecha_envio DESC LIMIT 1) as fecha_ultimo_mensaje,
                    (SELECT u.nombre FROM usuarios u JOIN memorandos_mensajes mm ON u.id = mm.remitente_id WHERE mm.memorando_id = m.id ORDER BY mm.fecha_envio DESC LIMIT 1) as remitente_ultimo_mensaje,
                    -- CAMBIO: Lógica corregida para contar mensajes no leídos, incluyendo chats nunca abiertos (fecha_lectura IS NULL)
                    (SELECT COUNT(*) FROM memorandos_mensajes mm WHERE mm.memorando_id = m.id AND (mp.fecha_lectura IS NULL OR mm.fecha_envio > mp.fecha_lectura)) as mensajes_no_leidos
                FROM memorandos m
                JOIN memorandos_participantes mp ON m.id = mp.memorando_id
                WHERE mp.usuario_id = ?
                ORDER BY fecha_ultimo_mensaje DESC
            ");
            $stmt->execute([$usuario_id]);
            $conversaciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $conversaciones];
            break;

        case 'cargar_mensajes':
            $memorando_id = filter_input(INPUT_GET, 'memorando_id', FILTER_VALIDATE_INT);
            if (!$memorando_id) { $response['message'] = 'ID de memorando no válido.'; break; }
            
            $stmt_check = $conn->prepare("SELECT COUNT(*) FROM memorandos_participantes WHERE memorando_id = ? AND usuario_id = ?");
            $stmt_check->execute([$memorando_id, $usuario_id]);
            if ($stmt_check->fetchColumn() == 0) { $response['message'] = 'No tienes acceso a esta conversación.'; break; }
            
            $stmt_info = $conn->prepare("SELECT tipo FROM memorandos WHERE id = ?");
            $stmt_info->execute([$memorando_id]);
            $tipo_memorando = $stmt_info->fetchColumn();
            
            // CAMBIO: Se limita la consulta a los últimos 50 mensajes para optimizar la carga. Se ordenan DESC para obtener los últimos.
            $stmt_mensajes = $conn->prepare("
                SELECT mm.id, mm.remitente_id, u.nombre as remitente_nombre, u.foto_perfil, mm.mensaje, mm.fecha_envio
                FROM memorandos_mensajes mm JOIN usuarios u ON mm.remitente_id = u.id
                WHERE mm.memorando_id = ? ORDER BY mm.fecha_envio DESC LIMIT 50
            ");
            $stmt_mensajes->execute([$memorando_id]);
            $mensajes = $stmt_mensajes->fetchAll(PDO::FETCH_ASSOC);
            $mensajes = array_reverse($mensajes); // CAMBIO: Se revierte el array para que el frontend los reciba en orden cronológico (ASC).

            // NUEVO: Lógica para añadir el estado de "leído".
            $stmt_participantes = $conn->prepare("SELECT usuario_id, fecha_lectura FROM memorandos_participantes WHERE memorando_id = ? AND usuario_id != ?");
            $stmt_participantes->execute([$memorando_id, $usuario_id]);
            $participantes = $stmt_participantes->fetchAll(PDO::FETCH_KEY_PAIR);

            if (!empty($participantes)) {
                foreach ($mensajes as &$mensaje) { // Usamos referencia (&) para modificar el array original
                    if ($mensaje['remitente_id'] == $usuario_id) {
                        $mensaje['leido_por_todos'] = true;
                        foreach ($participantes as $fecha_lectura_participante) {
                            if ($fecha_lectura_participante === null || strtotime($fecha_lectura_participante) < strtotime($mensaje['fecha_envio'])) {
                                $mensaje['leido_por_todos'] = false;
                                break; // Si uno no lo ha leído, es suficiente.
                            }
                        }
                    }
                }
                unset($mensaje); // Romper la referencia
            }
            
            $stmt_leido = $conn->prepare("UPDATE memorandos_participantes SET fecha_lectura = NOW() WHERE memorando_id = ? AND usuario_id = ?");
            $stmt_leido->execute([$memorando_id, $usuario_id]);
            
            $response = ['success' => true, 'data' => ['mensajes' => $mensajes, 'tipo' => $tipo_memorando]];
            break;

        case 'enviar_mensaje':
            $memorando_id = filter_input(INPUT_POST, 'memorando_id', FILTER_VALIDATE_INT);
            $mensaje = trim($_POST['mensaje'] ?? '');
            if (!$memorando_id || empty($mensaje)) { $response['message'] = 'Faltan datos para enviar el mensaje.'; break; }
            
            $stmt_check_tipo = $conn->prepare("SELECT tipo FROM memorandos WHERE id = ?");
            $stmt_check_tipo->execute([$memorando_id]);
            if ($stmt_check_tipo->fetchColumn() === 'Comunicado') { $response['message'] = 'No se puede responder a un comunicado.'; break; }
            
            $stmt_check = $conn->prepare("SELECT COUNT(*) FROM memorandos_participantes WHERE memorando_id = ? AND usuario_id = ?");
            $stmt_check->execute([$memorando_id, $usuario_id]);
            if ($stmt_check->fetchColumn() == 0) { $response['message'] = 'No puedes enviar mensajes en esta conversación.'; break; }
            
            $conn->beginTransaction();
            $stmt_insert = $conn->prepare("INSERT INTO memorandos_mensajes (memorando_id, remitente_id, mensaje) VALUES (?, ?, ?)");
            $stmt_insert->execute([$memorando_id, $usuario_id, $mensaje]);
            
            // CAMBIO: Se actualiza la fecha de lectura. Se establece a AHORA para el remitente y se resetea a NULL para todos los demás participantes.
            $stmt_update_leidos = $conn->prepare("
                UPDATE memorandos_participantes 
                SET fecha_lectura = IF(usuario_id = ?, NOW(), NULL) 
                WHERE memorando_id = ?
            ");
            $stmt_update_leidos->execute([$usuario_id, $memorando_id]);
            
            $conn->commit();
            $response = ['success' => true, 'message' => 'Mensaje enviado.'];
            break;

        case 'crear_conversacion':
            $asunto = trim($_POST['asunto'] ?? '');
            $mensaje = trim($_POST['mensaje'] ?? '');
            $destinatarios_ids = isset($_POST['destinatarios']) ? json_decode($_POST['destinatarios'], true) : [];
            $es_comunicado = (isset($_POST['es_comunicado']) && $_POST['es_comunicado'] === 'true');
            $tipo_memorando = $es_comunicado ? 'Comunicado' : 'Normal';

            if (empty($asunto) || empty($mensaje)) { $response['message'] = 'Asunto y mensaje son requeridos.'; break; }
            if (!$es_comunicado && empty($destinatarios_ids)) { $response['message'] = 'Debe seleccionar al menos un destinatario.'; break; }

            $conn->beginTransaction();

            if ($es_comunicado) {
                $stmt_todos = $conn->query("SELECT id FROM usuarios WHERE estado = 1");
                $destinatarios_ids = $stmt_todos->fetchAll(PDO::FETCH_COLUMN);
            } else {
                 if (!in_array($usuario_id, $destinatarios_ids)) { $destinatarios_ids[] = (string)$usuario_id; }
            }

            $stmt_memo = $conn->prepare("INSERT INTO memorandos (asunto, creado_por_id, tipo) VALUES (?, ?, ?)");
            $stmt_memo->execute([$asunto, $usuario_id, $tipo_memorando]);
            $memorando_id = $conn->lastInsertId();
            
            $stmt_msg = $conn->prepare("INSERT INTO memorandos_mensajes (memorando_id, remitente_id, mensaje) VALUES (?, ?, ?)");
            $stmt_msg->execute([$memorando_id, $usuario_id, $mensaje]);

            $stmt_part = $conn->prepare("INSERT INTO memorandos_participantes (memorando_id, usuario_id, fecha_lectura) VALUES (?, ?, ?)");
            foreach (array_unique($destinatarios_ids) as $dest_id) {
                // Solo el creador tiene fecha de lectura al inicio. El resto NULL.
                $fecha_lectura = ($dest_id == $usuario_id) ? date('Y-m-d H:i:s') : null;
                $stmt_part->execute([$memorando_id, (int)$dest_id, $fecha_lectura]);
            }
            
            $conn->commit();
            $response = ['success' => true, 'message' => 'Memorando enviado correctamente.'];
            break;

        // El caso 'verificar_notificaciones' puede ser obsoleto con la nueva lógica, pero lo mantenemos por si se usa en otra parte.
        case 'verificar_notificaciones':
             $stmt = $conn->prepare("
                SELECT COUNT(DISTINCT m.id)
                FROM memorandos m
                JOIN memorandos_participantes mp ON m.id = mp.memorando_id
                JOIN memorandos_mensajes mm ON m.id = mm.memorando_id
                WHERE mp.usuario_id = ? AND (mp.fecha_lectura IS NULL OR mm.fecha_envio > mp.fecha_lectura)
            ");
            $stmt->execute([$usuario_id]);
            $count = $stmt->fetchColumn();
            $response = ['success' => true, 'data' => ['nuevos_mensajes' => (int)$count > 0]];
            break;

        default:
            $response['message'] = 'Acción de mensajería desconocida.';
            break;
    }
} catch (PDOException $e) {
    if ($conn->inTransaction()) { $conn->rollBack(); }
    error_log("Error en mensajes_ajax.php: " . $e->getMessage());
    $response['message'] = 'Error de base de datos: ' . $e->getMessage(); // Más detallado para depuración
}

echo json_encode($response);