<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../administracion/includes/seguridad.php';
require_once 'lib/fpdf.php';

class PDF_Reporte_Ventas extends FPDF {
    private $fecha_inicio;
    private $fecha_fin;

    function __construct($fecha_inicio, $fecha_fin) {
        parent::__construct('P', 'mm', 'Letter');
        $this->fecha_inicio = $fecha_inicio;
        $this->fecha_fin = $fecha_fin;
    }
    
    function TextoUTF8($texto) {
        return iconv('UTF-8', 'ISO-8859-1//TRANSLIT', $texto);
    }
    
    function Header() {
        if (file_exists('../assets/images/logo.png')) {
            $this->Image('../assets/images/logo.png', 10, 8, 25);
        }
        $this->SetFont('Arial','B',16);
        $this->SetTextColor(27, 94, 32);
        $this->Cell(0,10,$this->TextoUTF8('REPORTE GENERAL DE VENTAS'),0,1,'C');
        $this->SetFont('Arial','',10);
        $this->SetTextColor(100);
        $fecha_inicio_f = date('d/m/Y', strtotime($this->fecha_inicio));
        $fecha_fin_f = date('d/m/Y', strtotime($this->fecha_fin));
        $this->Cell(0,5,$this->TextoUTF8("Periodo del {$fecha_inicio_f} al {$fecha_fin_f}"),0,1,'C');
        $this->Ln(10);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetFont('Arial','I',8);
        $this->Cell(0,10,$this->TextoUTF8('Página ').$this->PageNo().'/{nb}',0,0,'C');
    }
    
    function SeccionVenta($venta) {
        $this->SetFillColor(230, 245, 230);
        $this->SetDrawColor(76, 175, 80);
        $this->SetFont('Arial','B',10);
        
        $this->Cell(40, 7, $this->TextoUTF8(' Venta #' . $venta['id']), 'LT', 0, 'L', true);
        $this->Cell(55, 7, $this->TextoUTF8('Fecha: ' . date('d/m/Y', strtotime($venta['fecha_venta']))), 'T', 0, 'L', true);
        $this->Cell(95, 7, $this->TextoUTF8('Estado: ' . $venta['estado']), 'TR', 1, 'R', true);
        
        $this->SetFillColor(255);
        $this->SetFont('Arial','',9);
        $this->Cell(190, 6, $this->TextoUTF8(' Cliente: ' . $venta['cliente_nombre'] . ' (C.I: ' . $venta['cedula'] . ')'), 'LR', 1, 'L');
        $this->Cell(190, 6, $this->TextoUTF8(' Asesor: ' . $venta['operador_nombre']), 'LRB', 1, 'L');

        $this->Ln(2);
        // Tabla de items
        $this->SetFont('Arial','B',9);
        $this->SetFillColor(76, 175, 80);
        $this->SetTextColor(255);
        $this->Cell(90, 6, 'Producto', 1, 0, 'C', true);
        $this->Cell(25, 6, 'Cant.', 1, 0, 'C', true);
        $this->Cell(35, 6, 'P. Unitario', 1, 0, 'C', true);
        $this->Cell(40, 6, 'Subtotal', 1, 1, 'C', true);
        
        $this->SetFont('Arial','',8);
        $this->SetTextColor(0);
        
        if (empty($venta['items'])) {
            $this->Cell(190, 6, $this->TextoUTF8('No se encontraron productos para esta venta.'), 1, 1, 'C');
        } else {
            foreach ($venta['items'] as $item) {
                $this->Cell(90, 5, $this->TextoUTF8(substr($item['producto_nombre'], 0, 45)), 'LR');
                $this->Cell(25, 5, $item['cantidad'], 'R', 0, 'C');
                $this->Cell(35, 5, '$ ' . number_format($item['precio_unitario'], 2, ',', '.'), 'R', 0, 'R');
                $this->Cell(40, 5, '$ ' . number_format($item['subtotal'], 2, ',', '.'), 'R', 1, 'R');
            }
        }

        $this->SetFont('Arial','B',9);
        $this->Cell(150, 6, 'Total Venta:', 'T', 0, 'R');
        $this->Cell(40, 6, '$ ' . number_format($venta['monto_total'], 2, ',', '.'), 'T', 1, 'R');
        $this->Ln(8);
    }
}

// ================= LÓGICA PRINCIPAL ==================

if (!puede('gestionar_despachos')) {
    die('Acceso denegado.');
}

$fecha_inicio = $_GET['fecha_inicio'] ?? date('Y-m-d');
$fecha_fin = $_GET['fecha_fin'] ?? date('Y-m-d');

try {
    // 1. Obtener todas las ventas en el rango de fechas
    $stmt_ventas = $conn->prepare("
        SELECT v.id, v.fecha_venta, v.monto_total, v.estado, c.nombre as cliente_nombre, c.cedula, u.nombre as operador_nombre
        FROM ventas v
        JOIN clientes c ON v.cliente_id = c.id
        JOIN usuarios u ON v.operador_id = u.id
        WHERE v.fecha_venta BETWEEN ? AND ?
        ORDER BY v.fecha_venta ASC, v.id ASC
    ");
    $stmt_ventas->execute([$fecha_inicio, $fecha_fin]);
    $ventas = $stmt_ventas->fetchAll(PDO::FETCH_ASSOC);

    // 2. Para cada venta, obtener sus items
    $stmt_items = $conn->prepare("
        SELECT vi.cantidad, vi.precio_unitario, vi.subtotal, p.nombre as producto_nombre
        FROM venta_items vi
        JOIN almacen_productos p ON vi.producto_id = p.id
        WHERE vi.venta_id = ?
    ");

    $total_general = 0;
    foreach ($ventas as &$venta) {
        $stmt_items->execute([$venta['id']]);
        $venta['items'] = $stmt_items->fetchAll(PDO::FETCH_ASSOC);
        if ($venta['estado'] !== 'Cancelado') {
            $total_general += $venta['monto_total'];
        }
    }

    // 3. Generar el PDF
    $pdf = new PDF_Reporte_Ventas($fecha_inicio, $fecha_fin);
    $pdf->AliasNbPages();
    $pdf->AddPage();

    if (empty($ventas)) {
        $pdf->SetFont('Arial', 'B', 12);
        $pdf->Cell(0, 20, $pdf->TextoUTF8('No se encontraron ventas en el período seleccionado.'), 0, 1, 'C');
    } else {
        foreach ($ventas as $venta) {
            // Verificar si hay espacio para la venta completa, si no, saltar de página
            if ($pdf->GetY() > 220) {
                $pdf->AddPage();
            }
            $pdf->SeccionVenta($venta);
        }

        // Resumen final
        $pdf->SetFont('Arial', 'B', 14);
        $pdf->SetFillColor(27, 94, 32);
        $pdf->SetTextColor(255);
        $pdf->Cell(130, 10, 'MONTO TOTAL GENERAL (Ventas No Canceladas):', 0, 0, 'R', true);
        $pdf->Cell(60, 10, '$ ' . number_format($total_general, 2, ',', '.'), 0, 1, 'C', true);
    }
    
    $pdf->Output('D', "Reporte_Ventas_{$fecha_inicio}_a_{$fecha_fin}.pdf");

} catch (Exception $e) {
    die("Error al generar el PDF: " . $e->getMessage());
}
?>