<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']))) {
    $response['message'] = 'Error de seguridad (CSRF).';
    echo json_encode($response);
    exit();
}

if (!puede('gestionar_despachos')) {
    $response['message'] = 'Acceso denegado.';
    echo json_encode($response);
    exit();
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'cargar_ventas':
        try {
            $stmt = $conn->query("
                SELECT v.id, v.fecha_venta, v.monto_total, v.estado,
                       c.nombre as cliente_nombre,
                       u.nombre as operador_nombre
                FROM ventas v
                JOIN clientes c ON v.cliente_id = c.id
                JOIN usuarios u ON v.operador_id = u.id
                ORDER BY v.fecha_creacion DESC
            ");
            $response = ['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
        } catch (PDOException $e) {
            $response['message'] = 'Error al cargar las ventas.';
        }
        break;

    case 'obtener_detalle_venta':
        $venta_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$venta_id) { $response['message'] = 'ID de venta no válido.'; break; }
        try {
            $stmt_venta = $conn->prepare("SELECT * FROM ventas WHERE id = ?");
            $stmt_venta->execute([$venta_id]);
            $venta = $stmt_venta->fetch(PDO::FETCH_ASSOC);

            if (!$venta) { $response['message'] = 'Venta no encontrada.'; break; }

            $stmt_cliente = $conn->prepare("SELECT nombre, cedula, telefono FROM clientes WHERE id = ?");
            $stmt_cliente->execute([$venta['cliente_id']]);
            $cliente = $stmt_cliente->fetch(PDO::FETCH_ASSOC);

            $stmt_items = $conn->prepare("
                SELECT vi.*, p.nombre as producto_nombre, p.sku
                FROM venta_items vi
                JOIN almacen_productos p ON vi.producto_id = p.id
                WHERE vi.venta_id = ?
            ");
            $stmt_items->execute([$venta_id]);
            $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => [
                'venta' => $venta,
                'items' => $items,
                'cliente' => $cliente
            ]];
        } catch (PDOException $e) {
            $response['message'] = 'Error al obtener los detalles.';
        }
        break;

    case 'actualizar_estado_venta':
        $venta_id = filter_input(INPUT_POST, 'venta_id', FILTER_VALIDATE_INT);
        $nuevo_estado = $_POST['nuevo_estado'] ?? '';
        $estados_validos = ['Verificado', 'En Despacho', 'Completado', 'Cancelado'];

        if (!$venta_id || !in_array($nuevo_estado, $estados_validos)) {
            $response['message'] = 'Datos inválidos para actualizar.';
            break;
        }

        try {
            $conn->beginTransaction();
            
            // Obtenemos el estado actual para usarlo en los registros de la bitácora
            $stmt_estado_actual = $conn->prepare("SELECT estado FROM ventas WHERE id = ?");
            $stmt_estado_actual->execute([$venta_id]);
            $estado_actual = $stmt_estado_actual->fetchColumn();

            // Variables para un registro de bitácora más detallado
            $accion_log = 'cambiar_estado_venta';
            $detalle_log = "Cambió el estado de la venta #{$venta_id} de '{$estado_actual}' a '{$nuevo_estado}'.";

            // --- LÓGICA DE MANEJO DE STOCK ---
            
            // Si la venta pasa de 'Pendiente' a 'Verificado', se descuenta el stock.
            if ($nuevo_estado === 'Verificado' && $estado_actual === 'Pendiente') {
                $stmt_items = $conn->prepare("SELECT producto_id, cantidad FROM venta_items WHERE venta_id = ?");
                $stmt_items->execute([$venta_id]);
                $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);
                $stmt_movimiento = $conn->prepare("INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, referencia_id, usuario_id) VALUES (?, 'egreso', ?, ?, ?, ?)");

                foreach ($items as $item) {
                    $stmt_stock = $conn->prepare("UPDATE almacen_stock SET cantidad = cantidad - ? WHERE producto_id = ? AND cantidad >= ?");
                    $stmt_stock->execute([$item['cantidad'], $item['producto_id'], $item['cantidad']]);
                    if ($stmt_stock->rowCount() == 0) {
                        throw new Exception("Stock insuficiente para el producto ID {$item['producto_id']}. No se puede verificar la venta.");
                    }
                    $motivo = "Venta Verificada #" . $venta_id;
                    $stmt_movimiento->execute([$item['producto_id'], $item['cantidad'], $motivo, $venta_id, $usuario_id]);
                }
                $accion_log = 'verificar_venta';
                $detalle_log = "Verificó la venta #{$venta_id}, descontando el stock correspondiente del inventario.";
            }
            
            // Si se cancela una venta que ya había afectado el stock ('Verificado' o 'En Despacho'), se repone.
            if ($nuevo_estado === 'Cancelado' && in_array($estado_actual, ['Verificado', 'En Despacho'])) {
                $stmt_items = $conn->prepare("SELECT producto_id, cantidad FROM venta_items WHERE venta_id = ?");
                $stmt_items->execute([$venta_id]);
                $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);
                $stmt_movimiento = $conn->prepare("INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, referencia_id, usuario_id) VALUES (?, 'ingreso', ?, ?, ?, ?)");

                foreach ($items as $item) {
                    $conn->prepare("UPDATE almacen_stock SET cantidad = cantidad + ? WHERE producto_id = ?")->execute([$item['cantidad'], $item['producto_id']]);
                    $motivo = "Venta Cancelada #" . $venta_id;
                    $stmt_movimiento->execute([$item['producto_id'], $item['cantidad'], $motivo, $venta_id, $usuario_id]);
                }
                $accion_log = 'cancelar_venta';
                $detalle_log = "Canceló la venta #{$venta_id} (estado anterior: '{$estado_actual}'), reponiendo los productos al inventario.";
            }
            
            // --- LÓGICA DE ACTUALIZACIÓN DE VENTA ---

            if ($nuevo_estado === 'En Despacho') {
                $empresa_envio = trim($_POST['empresa_envio'] ?? '');
                $numero_guia = trim($_POST['numero_guia'] ?? '');
                if(empty($empresa_envio) || empty($numero_guia)){
                    throw new Exception("Debe seleccionar una empresa de envío e ingresar el número de guía.");
                }
                $stmt_update = $conn->prepare("UPDATE ventas SET estado = ?, empresa_envio = ?, numero_guia = ? WHERE id = ?");
                $stmt_update->execute([$nuevo_estado, $empresa_envio, $numero_guia, $venta_id]);

                // Log específico para el despacho
                $accion_log = 'despachar_venta';
                $detalle_log = "Registró el despacho de la venta #{$venta_id} por {$empresa_envio} con guía {$numero_guia}.";
            } else {
                // Para todos los demás cambios de estado
                $stmt_update = $conn->prepare("UPDATE ventas SET estado = ? WHERE id = ?");
                $stmt_update->execute([$nuevo_estado, $venta_id]);
                
                // Log específico si se cancela sin afectar stock
                if ($nuevo_estado === 'Cancelado' && !in_array($estado_actual, ['Verificado', 'En Despacho'])) {
                    $accion_log = 'cancelar_venta';
                    $detalle_log = "Canceló la venta #{$venta_id} (estado anterior: '{$estado_actual}'). No hubo afectación de inventario.";
                }
            }

            $conn->commit();
            // Se registra la acción detallada que corresponda
            registrar_accion($accion_log, $detalle_log);
            $response = ['success' => true, 'message' => "Estado de la venta actualizado a '{$nuevo_estado}'."];

        } catch (Exception $e) {
            if ($conn->inTransaction()) {
                $conn->rollBack();
            }
            $response['message'] = $e->getMessage();
        }
        break;
}

echo json_encode($response);