<?php
require_once 'includes/seguridad.php';

// 1. Validar que el usuario tenga permisos
if (!puede('gestionar_nomina')) {
    header('Location: inicio.php?error=acceso_denado');
    exit();
}

// 2. Validar que se reciba un ID numérico por la URL
$nomina_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$nomina_id) {
    header('Location: nominas.php?error=id_invalido');
    exit();
}

$page_title = 'Editar Nómina #' . $nomina_id;
include_once 'templates/header.php';
?>

<style>
    .empleados-list { max-height: 250px; overflow-y: auto; border: 1px solid var(--border-color); border-radius: .375rem; }
    .empleado-item { display: flex; justify-content: space-between; align-items: center; padding: 0.5rem 0.75rem; border-bottom: 1px solid var(--border-color); }
    .empleado-item:last-child { border-bottom: none; }
    .ami-accordion .accordion-item { background-color: transparent; border: 1px solid var(--border-color); margin-bottom: 0.5rem; border-radius: .375rem !important; overflow: hidden; }
    .ami-accordion .accordion-header { position: relative; }
    .ami-accordion .accordion-button { background-color: rgba(0,0,0,0.2) !important; color: var(--light-text) !important; font-weight: bold; }
    .ami-accordion .accordion-button:not(.collapsed) { background-color: var(--primary-color) !important; color: var(--dark-text) !important; box-shadow: none; }
    .ami-accordion .accordion-button::after { filter: invert(1) grayscale(100%) brightness(200%); }
    .ami-accordion .accordion-body { background-color: rgba(0,0,0,0.1); padding: 1.25rem; }
    .ami-accordion .resumen-colapsado { display: flex; justify-content: space-between; width: 100%; padding-right: 3rem; }
    .ami-accordion .accordion-button.collapsed .resumen-colapsado .text-primary, .ami-accordion .accordion-button.collapsed .resumen-colapsado .text-success { color: var(--light-text) !important; }
    .ami-accordion .text-success { color: #20c997 !important; }
    .btn-remove-from-nomina { position: absolute; right: 3.25rem; top: 50%; transform: translateY(-50%); z-index: 5; background: #dc3545; color: white; border: none; border-radius: 50%; width: 24px; height: 24px; font-size: 14px; display: flex; align-items: center; justify-content: center; font-weight: bold; line-height: 1; opacity: 0.7; transition: opacity 0.2s, transform 0.2s; }
    .btn-remove-from-nomina:hover { opacity: 1; transform: translateY(-50%) scale(1.1); }
    .calc-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1.25rem; }
    .daily-calc-box { background-color: rgba(0,0,0,0.2); border: 1px solid var(--border-color); padding: 0.375rem 0.75rem; border-radius: .375rem; font-weight: bold; color: var(--primary-color); }
    .resumen-linea { display: flex; justify-content: space-between; padding: 0.25rem 0; font-size: 1.1em; color: ghostwhite; }
    .text-muted { --bs-text-opacity: 1; color: lawngreen !important; }
    h6 { color: white; }
</style>

<div class="main-header">
    <a href="nominas.php" class="btn-back"><i class="fa-solid fa-arrow-left"></i> Volver</a>
    <h1><?php echo htmlspecialchars($page_title); ?></h1>
</div>

<div class="info-card">
    <div id="loading-overlay" class="text-center p-5"><span class="spinner-border"></span> Cargando datos de la nómina...</div>

    <form id="form-nomina" class="d-none">
        <div class="row mb-4">
            <div class="col-lg-5">
                <h5 class="detalle-titulo">1. Datos Generales</h5>
                <div class="mb-3">
                    <label class="ami-form-label">Nombre de la Nómina</label>
                    <input type="text" id="nombre_nomina" class="form-control" placeholder="Ej: 1ra Quincena Septiembre">
                </div>
                <div class="row">
                    <div class="col-md-6 mb-3 mb-md-0"><label class="ami-form-label">Fecha Inicio</label><input type="date" id="fecha_inicio" class="form-control"></div>
                    <div class="col-md-6"><label class="ami-form-label">Fecha Fin</label><input type="date" id="fecha_fin" class="form-control"></div>
                </div>
                <div class="mt-3">
                    <label class="ami-form-label">Tasa de Cambio (BCV)</label>
                    <div class="input-group"><span class="input-group-text">Bs.</span><input type="number" step="0.01" id="tasa_cambio" class="form-control" placeholder="0.00"></div>
                </div>
            </div>
            <div class="col-lg-7">
                <h5 class="detalle-titulo">2. Añadir Empleados a la Nómina</h5>
                <input type="text" id="search-empleado" class="form-control mb-2" placeholder="Buscar empleado por nombre o cédula...">
                <div class="empleados-list" id="lista-empleados-disponibles"></div>
            </div>
        </div>
        
        <div class="d-flex justify-content-between align-items-center">
            <h5 class="detalle-titulo mb-0">3. Cálculo de Pagos</h5>
            <button type="button" class="btn btn-sm btn-outline-secondary" id="toggle-all-cards">Expandir/Colapsar Todos</button>
        </div>
        <hr>
        <div class="accordion ami-accordion" id="area-calculo-nomina">
            <p class="text-muted text-center p-5" id="mensaje-inicial">Añada empleados desde el panel superior para comenzar.</p>
        </div>

        <div class="text-end mt-4">
            <a href="nominas.php" class="btn btn-secondary"><i class="fa-solid fa-times me-2"></i>Cancelar</a>
            <button type="button" class="btn btn-primary" id="btn-guardar-nomina">
                <i class="fa-solid fa-save me-1"></i> Guardar Cambios
            </button>
        </div>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';
    const nominaId = <?php echo $nomina_id; ?>;
    let todosLosEmpleados = [];
    let empleadosEnNomina = new Set();
    const DIAS_MES = 30;

    const UI = {
        form: document.getElementById('form-nomina'),
        loadingOverlay: document.getElementById('loading-overlay'),
        listaDisponibles: document.getElementById('lista-empleados-disponibles'),
        areaCalculo: document.getElementById('area-calculo-nomina'),
        searchInput: document.getElementById('search-empleado'),
        tasaCambioInput: document.getElementById('tasa_cambio'),
        btnGuardar: document.getElementById('btn-guardar-nomina'),
        mensajeInicial: document.getElementById('mensaje-inicial'),
        btnToggleAll: document.getElementById('toggle-all-cards')
    };

    const formatoMoneda = (valor, simbolo = '$') => `${simbolo}${ (parseFloat(valor) || 0).toFixed(2) }`;
    const formatoBs = (valor) => `Bs. ${ (parseFloat(valor) || 0).toLocaleString('es-VE', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) }`;

    const calcularTarjeta = (accordionItem) => {
        const getVal = selector => parseFloat(accordionItem.querySelector(`[data-field="${selector}"]`).value) || 0;
        const setCalc = (selector, value, formato) => accordionItem.querySelector(`[data-calc="${selector}"]`).textContent = formato(value);
        
        const tasaCambio = parseFloat(UI.tasaCambioInput.value) || 0;
        const sueldoBase = getVal('sueldo_base');
        setCalc('salario_diario_usd', sueldoBase / DIAS_MES, v => v.toFixed(2));
        const cestaTicket = getVal('cesta_ticket');
        setCalc('cesta_ticket_diario_usd', cestaTicket / DIAS_MES, v => v.toFixed(2));
        const compCT = getVal('complemento_cesta_ticket');
        setCalc('comp_cesta_ticket_diario_usd', compCT / DIAS_MES, v => v.toFixed(2));
        const bonoTransp = getVal('bono_transporte');
        setCalc('bono_transporte_diario_usd', bonoTransp / DIAS_MES, v => v.toFixed(2));
        const compProd = getVal('complemento_productividad');

        const totalAsignaciones = sueldoBase + cestaTicket + compCT + compProd + bonoTransp;
        const abonoQuincenal = totalAsignaciones / 2;
        // --- CORRECCIÓN EN CÁLCULO DE DEDUCCIONES ---
        // Se alinea con la lógica de nomina-crear.php
        const totalDeducciones = getVal('adelantos') + getVal('dias_no_laborados') + getVal('deducciones');
        const totalPagarUSD = abonoQuincenal - totalDeducciones;
        const totalPagarBS = totalPagarUSD * tasaCambio;

        setCalc('total_asignaciones', totalAsignaciones, formatoMoneda);
        setCalc('abono_quincenal', abonoQuincenal, formatoMoneda);
        setCalc('total_deducciones', totalDeducciones, formatoMoneda);
        setCalc('total_pagar_resumen', totalPagarUSD, formatoMoneda);
        setCalc('total_pagar_bs_resumen', totalPagarBS, formatoBs);
        setCalc('total_pagar_expandido', totalPagarUSD, formatoMoneda);
        setCalc('total_pagar_bs_expandido', totalPagarBS, formatoBs);
    };

    const renderizarListaDisponibles = () => {
        const filtro = UI.searchInput.value.toLowerCase();
        UI.listaDisponibles.innerHTML = '';
        const fragment = document.createDocumentFragment();
        todosLosEmpleados.forEach(emp => {
            if (!empleadosEnNomina.has(parseInt(emp.id)) && (emp.nombre.toLowerCase().includes(filtro) || emp.cedula.includes(filtro))) {
                const item = document.createElement('div');
                item.className = 'empleado-item';
                item.innerHTML = `<div><strong>${emp.nombre}</strong><br><small class="text-muted">C.I: ${emp.cedula}</small></div><button type="button" class="btn btn-sm btn-success btn-add-empleado" data-id="${emp.id}" title="Añadir a la nómina">+</button>`;
                fragment.appendChild(item);
            }
        });
        UI.listaDisponibles.appendChild(fragment);
    };

    const añadirEmpleadoACalculo = (id, detalle = null) => {
        const empleado = todosLosEmpleados.find(e => e.id == id);
        if (!empleado) return;

        UI.mensajeInicial.style.display = 'none';
        const collapseId = `collapse-${empleado.id}`;
        const item = document.createElement('div');
        item.className = 'accordion-item';
        item.dataset.empleadoId = empleado.id;
        // --- CORRECCIÓN EN ESTRUCTURA HTML DE LA TARJETA ---
        // Se alinea con los campos de nomina-crear.php (ej. 'dias_no_laborados', 'deducciones')
        item.innerHTML = `
            <h2 class="accordion-header">
                <button type="button" class="btn-remove-from-nomina" title="Quitar de la nómina">&times;</button>
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#${collapseId}">
                    <div class="resumen-colapsado">
                        <span>${empleado.nombre}</span>
                        <span>
                            <strong class="text-primary me-3" data-calc="total_pagar_resumen">$0.00</strong>
                            <strong class="text-success" data-calc="total_pagar_bs_resumen">Bs. 0,00</strong>
                        </span>
                    </div>
                </button>
            </h2>
            <div id="${collapseId}" class="accordion-collapse collapse" data-bs-parent="#area-calculo-nomina">
                <div class="accordion-body">
                    <h6 class="detalle-titulo">Asignaciones Mensuales ($)</h6>
                    <div class="calc-grid">
                        <div><label class="ami-form-label">Sueldo Base</label><input type="number" step="0.01" class="form-control" data-field="sueldo_base" value="${parseFloat(detalle ? detalle.sueldo_base : empleado.salario_base).toFixed(2)}"></div>
                        <div><label class="ami-form-label">Cesta Ticket</label><input type="number" step="0.01" class="form-control" data-field="cesta_ticket" value="${parseFloat(detalle ? detalle.cesta_ticket : empleado.cesta_ticket).toFixed(2)}"></div>
                        <div><label class="ami-form-label">Comp. C.T.</label><input type="number" step="0.01" class="form-control" data-field="complemento_cesta_ticket" value="${parseFloat(detalle ? detalle.complemento_cesta_ticket : empleado.complemento_cesta_ticket).toFixed(2)}"></div>
                        <div><label class="ami-form-label">Bono Transp.</label><input type="number" step="0.01" class="form-control" data-field="bono_transporte" value="${parseFloat(detalle ? detalle.bono_transporte : empleado.bono_transporte).toFixed(2)}"></div>
                        <div><label class="ami-form-label">Comp. Prod.</label><input type="number" step="0.01" class="form-control" data-field="complemento_productividad" value="${parseFloat(detalle ? detalle.complemento_productividad : empleado.complemento_productividad).toFixed(2)}"></div>
                    </div><hr>
                    <div class="row">
                        <div class="col-lg-7">
                            <h6 class="detalle-titulo">Cálculos Diarios ($)</h6>
                            <div class="calc-grid">
                                <div><label class="ami-form-label">Salario Diario</label><p class="daily-calc-box" data-calc="salario_diario_usd">0.00</p></div>
                                <div><label class="ami-form-label">Cesta T. Diario</label><p class="daily-calc-box" data-calc="cesta_ticket_diario_usd">0.00</p></div>
                                <div><label class="ami-form-label">Comp. C.T. Diario</label><p class="daily-calc-box" data-calc="comp_cesta_ticket_diario_usd">0.00</p></div>
                                <div><label class="ami-form-label">Bono T. Diario</label><p class="daily-calc-box" data-calc="bono_transporte_diario_usd">0.00</p></div>
                            </div>
                        </div>
                        <div class="col-lg-5">
                            <h6 class="detalle-titulo">Deducciones ($)</h6>
                            <div><label class="ami-form-label">Días no Laborados</label><input type="number" step="0.01" class="form-control" data-field="dias_no_laborados" value="${parseFloat(detalle ? detalle.dias_no_laborados : '0.00').toFixed(2)}"></div>
                            <div class="mt-2"><label class="ami-form-label">Adelantos</label><input type="number" step="0.01" class="form-control" data-field="adelantos" value="${parseFloat(detalle ? detalle.adelantos : '0.00').toFixed(2)}"></div>
                            <div class="mt-2"><label class="ami-form-label">Otras Deduc.</label><input type="number" step="0.01" class="form-control" data-field="deducciones" value="${parseFloat(detalle ? detalle.deducciones : '0.00').toFixed(2)}"></div>
                        </div>
                    </div><hr>
                    <h6 class="detalle-titulo">Resumen Quincenal</h6>
                    <div class="resumen-linea"><span>Total Asignaciones (Mensual):</span> <strong data-calc="total_asignaciones">$0.00</strong></div>
                    <div class="resumen-linea"><span>Abono Quincenal (Asig / 2):</span> <strong><span class="fs-5" data-calc="abono_quincenal">$0.00</span></strong></div>
                    <div class="resumen-linea"><span>Total Deducciones:</span> <strong class="text-danger" data-calc="total_deducciones">$0.00</strong></div>
                    <hr>
                    <div class="text-end">
                        <div class="resumen-linea"><span>Total a Pagar ($):</span> <strong class="text-primary fs-4" data-calc="total_pagar_expandido">$0.00</strong></div>
                        <div class="resumen-linea"><span>Total a Pagar (Bs.):</span> <strong class="text-success fs-4" data-calc="total_pagar_bs_expandido">Bs. 0,00</strong></div>
                    </div>
                </div>
            </div>`;
        UI.areaCalculo.appendChild(item);
        empleadosEnNomina.add(parseInt(empleado.id));
        renderizarListaDisponibles();
        calcularTarjeta(item);
    };
    
    // --- Event Listeners (Sin cambios) ---
    UI.searchInput.addEventListener('input', renderizarListaDisponibles);
    UI.listaDisponibles.addEventListener('click', e => {
        if (e.target.classList.contains('btn-add-empleado')) añadirEmpleadoACalculo(e.target.dataset.id);
    });
    UI.areaCalculo.addEventListener('click', e => {
        if (e.target.classList.contains('btn-remove-from-nomina')) {
            const accordionItem = e.target.closest('.accordion-item');
            const idToRemove = parseInt(accordionItem.dataset.empleadoId);
            empleadosEnNomina.delete(idToRemove);
            accordionItem.remove();
            renderizarListaDisponibles();
            if (empleadosEnNomina.size === 0) UI.mensajeInicial.style.display = 'block';
        }
    });
    UI.areaCalculo.addEventListener('input', e => {
        if (e.target.matches('input[type="number"]')) calcularTarjeta(e.target.closest('.accordion-item'));
    });
    UI.tasaCambioInput.addEventListener('input', () => document.querySelectorAll('.accordion-item').forEach(calcularTarjeta));
    UI.btnToggleAll.addEventListener('click', () => {
        const allButtons = UI.areaCalculo.querySelectorAll('.accordion-button');
        const isAnyExpanded = Array.from(allButtons).some(btn => !btn.classList.contains('collapsed'));
        allButtons.forEach(btn => {
            const collapseEl = document.getElementById(btn.getAttribute('data-bs-target').substring(1));
            const bsCollapse = bootstrap.Collapse.getInstance(collapseEl) || new bootstrap.Collapse(collapseEl, { toggle: false });
            isAnyExpanded ? bsCollapse.hide() : bsCollapse.show();
        });
    });
    
    // --- Lógica de Guardado ---
    UI.btnGuardar.addEventListener('click', function() {
        // ... (Validaciones iniciales sin cambios)
        const btn = this;
        const nombreNomina = document.getElementById('nombre_nomina').value.trim();
        const fechaInicio = document.getElementById('fecha_inicio').value;
        const fechaFin = document.getElementById('fecha_fin').value;
        const tasaCambio = parseFloat(UI.tasaCambioInput.value) || 0;

        if (!nombreNomina || !fechaInicio || !fechaFin || tasaCambio <= 0) {
            Swal.fire('Datos Incompletos', 'Asegúrate de rellenar el nombre, las fechas y la tasa de cambio.', 'warning');
            return;
        }
        if (empleadosEnNomina.size === 0) {
            Swal.fire('Sin Empleados', 'Debes añadir al menos un empleado a la nómina.', 'warning');
            return;
        }

        let detalles = [];
        // --- CORRECCIÓN TOTAL EN LA RECOLECCIÓN DE DATOS ---
        // Se copia y pega la lógica exacta de nomina-crear.php
        document.querySelectorAll('.accordion-item').forEach(tarjeta => {
            const getVal = (selector) => {
                const input = tarjeta.querySelector(`[data-field="${selector}"]`);
                return input ? (parseFloat(input.value) || 0).toFixed(2) : '0.00';
            };
            
            const getCalc = (selector) => {
                const elemento = tarjeta.querySelector(`[data-calc="${selector}"]`);
                if (!elemento) return '0.00';
                
                let texto = elemento.textContent.trim();
                
                if (texto.includes('Bs.')) {
                    texto = texto.replace('Bs.', '').trim().replace(/\./g, '').replace(',', '.');
                } else if (texto.includes('$')) {
                    texto = texto.replace('$', '').trim().replace(/,/g, '');
                } else {
                    texto = texto.replace(',', '.');
                }
                
                const numero = parseFloat(texto);
                if (isNaN(numero)) {
                    console.warn(`Error al parsear valor de ${selector}: "${elemento.textContent}" -> "${texto}"`);
                    return '0.00';
                }
                
                return selector.includes('diario') ? numero.toFixed(4) : numero.toFixed(2);
            };
            
            detalles.push({
                empleado_id: tarjeta.dataset.empleadoId,
                sueldo_base: getVal('sueldo_base'),
                cesta_ticket: getVal('cesta_ticket'),
                complemento_cesta_ticket: getVal('complemento_cesta_ticket'),
                complemento_productividad: getVal('complemento_productividad'),
                bono_transporte: getVal('bono_transporte'),
                adelantos: getVal('adelantos'),
                dias_no_laborados: getVal('dias_no_laborados'),
                deducciones: getVal('deducciones'),
                salario_diario_usd: getCalc('salario_diario_usd'),
                cesta_ticket_diario_usd: getCalc('cesta_ticket_diario_usd'),
                comp_cesta_ticket_diario_usd: getCalc('comp_cesta_ticket_diario_usd'),
                bono_transporte_diario_usd: getCalc('bono_transporte_diario_usd'),
                total_asignaciones: getCalc('total_asignaciones'),
                total_deducciones: getCalc('total_deducciones'),
                total_pagar: getCalc('total_pagar_resumen'),
                total_pagar_bs: getCalc('total_pagar_bs_resumen')
            });
        });

        btn.disabled = true;
        btn.innerHTML = `<span class="spinner-border spinner-border-sm"></span> Guardando...`;

        const formData = new FormData();
        formData.append('action', 'editar_nomina');
        formData.append('csrf_token', csrfToken);
        formData.append('nomina_id', nominaId);
        formData.append('nombre_nomina', nombreNomina);
        formData.append('fecha_inicio', fechaInicio);
        formData.append('fecha_fin', fechaFin);
        formData.append('tasa_cambio', tasaCambio);
        formData.append('detalles', JSON.stringify(detalles));

        fetch('../api/nominas_ajax.php', { method: 'POST', body: formData })
            .then(res => res.json())
            .then(response => {
                if (response.success) {
                    Swal.fire({ icon: 'success', title: '¡Éxito!', text: response.message, timer: 1500, showConfirmButton: false })
                         .then(() => window.location.href = `nomina-detalle.php?id=${nominaId}`);
                } else Swal.fire('Error', response.message, 'error');
            })
            .catch(err => Swal.fire('Error de Conexión', 'No se pudo comunicar con el servidor.', 'error'))
            .finally(() => {
                btn.disabled = false;
                btn.innerHTML = '<i class="fa-solid fa-save me-1"></i> Guardar Cambios';
            });
    });

    // --- Carga Inicial de Datos (Sin cambios) ---
    fetch(`../api/nominas_ajax.php?action=obtener_datos_para_edicion&id=${nominaId}`)
        .then(res => res.json())
        .then(response => {
            if (response.success) {
                const { nomina, detalles_nomina, empleados } = response.data;

                document.getElementById('nombre_nomina').value = nomina.nombre_nomina;
                document.getElementById('fecha_inicio').value = nomina.fecha_inicio;
                document.getElementById('fecha_fin').value = nomina.fecha_fin;
                document.getElementById('tasa_cambio').value = parseFloat(nomina.tasa_cambio).toFixed(2);
                
                todosLosEmpleados = empleados;

                detalles_nomina.forEach(detalle => {
                    añadirEmpleadoACalculo(detalle.empleado_id, detalle);
                });
                
                renderizarListaDisponibles();

                UI.loadingOverlay.classList.add('d-none');
                UI.form.classList.remove('d-none');
            } else {
                UI.loadingOverlay.innerHTML = `<div class="alert alert-danger">${response.message}</div>`;
            }
        })
        .catch(err => {
            UI.loadingOverlay.innerHTML = `<div class="alert alert-danger">Error de conexión al cargar los datos.</div>`;
        });
});
</script>

<?php include_once 'templates/footer.php'; ?>