<?php
require_once 'includes/seguridad.php';
if (!puede('gestionar_despachos')) {
    header('Location: inicio.php?error=acceso_denegado');
    exit();
}
$page_title = 'Gestión de Despachos';
include_once 'templates/header.php';
?>

<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">

<style>
    .estado-pendiente { color: #ffc107; background-color: rgba(255, 193, 7, 0.1); border: 1px solid rgba(255, 193, 7, 0.2); }
    .estado-verificado { color: #0dcaf0; background-color: rgba(13, 202, 240, 0.1); border: 1px solid rgba(13, 202, 240, 0.2); }
    .estado-en-despacho { color: #6610f2; background-color: rgba(102, 16, 242, 0.1); border: 1px solid rgba(102, 16, 242, 0.2); }
    .estado-completado { color: #198754; background-color: rgba(25, 135, 84, 0.1); border: 1px solid rgba(25, 135, 84, 0.2); }
    .estado-cancelado { color: #dc3545; background-color: rgba(220, 53, 69, 0.1); border: 1px solid rgba(220, 53, 69, 0.2); }
    #venta-detalle-body h4 {
        color: var(--primary-color);
        border-bottom: 1px solid var(--border-color);
        padding-bottom: 0.5rem;
        margin-bottom: 1rem;
        font-size: 1.2rem;
    }
    #venta-detalle-body p { margin-bottom: 0.75rem; }
    #shipping-info-container {
        background-color: rgba(13, 202, 240, 0.1);
        border: 1px solid var(--border-color);
        border-radius: 8px;
        padding: 1.5rem;
        margin-top: 1rem;
    }
    @media (min-width: 1200px) {
      .modal-xl {
        max-width: 90vw !important; /* Ocupará el 90% del ancho de la pantalla */
      }
    }
</style>

<div class="main-header">
    <h1><i class="fa-solid fa-truck-fast text-primary"></i> <?php echo $page_title; ?></h1>
    <button class="btn-add-new" id="btnAbrirModalReporte"><i class="fa-solid fa-file-pdf"></i> Generar Reporte</button>
</div>

<div class="info-card">
    <div class="toolbar mb-3">
        <input type="text" id="searchInput" class="form-control form-control-dark" placeholder="Buscar por N° Venta, Cliente, Operador o Estado...">
    </div>
    <div class="table-responsive">
        <table class="ami-table">
            <thead>
                <tr>
                    <th>N° Venta</th>
                    <th>Fecha</th>
                    <th>Cliente</th>
                    <th>Operador</th>
                    <th class="text-end">Monto Total</th>
                    <th class="text-center">Estado</th>
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody id="tabla-despachos"></tbody>
        </table>
    </div>
</div>

<div class="modal fade ami-modal-theme" id="modalDetalleVenta" tabindex="-1">
  <div class="modal-dialog modal-xl"> 
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Detalles de la Venta #<span id="venta-id-modal"></span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body" id="venta-detalle-body"></div>
      <div class="modal-footer" id="venta-detalle-footer"></div>
    </div>
  </div>
</div>

<div class="modal fade ami-modal-theme" id="modalReporteFechas" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Generar Reporte de Ventas</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p>Selecciona el rango de fechas para generar el reporte en PDF.</p>
        <div class="ami-form-group mb-3">
            <label class="ami-form-label">Fecha de Inicio</label>
            <input type="text" id="reporte-fecha-inicio" class="form-control">
        </div>
        <div class="ami-form-group mb-3">
            <label class="ami-form-label">Fecha de Fin</label>
            <input type="text" id="reporte-fecha-fin" class="form-control">
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
        <button type="button" class="btn btn-primary" id="btnGenerarReportePdf"><i class="fa-solid fa-file-arrow-down"></i> Generar</button>
      </div>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
<script src="https://npmcdn.com/flatpickr/dist/l10n/es.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';
    const modalDetalle = new bootstrap.Modal(document.getElementById('modalDetalleVenta'));
    const modalReporte = new bootstrap.Modal(document.getElementById('modalReporteFechas'));
    let ventasData = [];

    // INICIALIZACIÓN DE SELECTORES DE FECHA
    const fpInicio = flatpickr("#reporte-fecha-inicio", { altInput: true, altFormat: "d/m/Y", dateFormat: "Y-m-d", locale: "es" });
    const fpFin = flatpickr("#reporte-fecha-fin", { altInput: true, altFormat: "d/m/Y", dateFormat: "Y-m-d", locale: "es" });

    function cargarVentas() {
        const tbody = document.getElementById('tabla-despachos');
        tbody.innerHTML = '<tr><td colspan="7" class="text-center">Cargando ventas...</td></tr>';
        fetch('/api/despachos_ajax.php?action=cargar_ventas')
            .then(res => res.json()).then(data => {
                if(data.success) {
                    ventasData = data.data;
                    renderizarTabla();
                } else {
                    tbody.innerHTML = `<tr><td colspan="7" class="text-center text-danger">${data.message}</td></tr>`;
                }
            });
    }

    function renderizarTabla() {
        const tbody = document.getElementById('tabla-despachos');
        const filtro = document.getElementById('searchInput').value.toLowerCase();
        const ventasFiltradas = ventasData.filter(v => 
            v.id.toString().includes(filtro) ||
            v.cliente_nombre.toLowerCase().includes(filtro) ||
            v.operador_nombre.toLowerCase().includes(filtro) ||
            v.estado.toLowerCase().includes(filtro)
        );
        if (ventasFiltradas.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No se encontraron ventas.</td></tr>';
            return;
        }
        tbody.innerHTML = ventasFiltradas.map(venta => {
            const fecha = new Date(venta.fecha_venta).toLocaleDateString('es-ES', { timeZone: 'UTC' });
            const estadoClass = `estado-${venta.estado.toLowerCase().replace(' ', '-')}`;
            return `
                <tr>
                    <td><strong>#${venta.id}</strong></td>
                    <td>${fecha}</td>
                    <td>${venta.cliente_nombre}</td>
                    <td>${venta.operador_nombre}</td>
                    <td class="text-end fw-bold">$${parseFloat(venta.monto_total).toFixed(2)}</td>
                    <td class="text-center"><span class="badge ${estadoClass}">${venta.estado}</span></td>
                    <td><button class="btn btn-sm btn-info" onclick="verDetalle(${venta.id})"><i class="fa-solid fa-eye"></i> Ver Detalles</button></td>
                </tr>`;
        }).join('');
    }

    window.verDetalle = function(ventaId) {
        document.getElementById('venta-id-modal').textContent = ventaId;
        document.getElementById('venta-detalle-body').innerHTML = '<p class="text-center">Cargando detalles...</p>';
        document.getElementById('venta-detalle-footer').innerHTML = '';
        
        fetch(`/api/despachos_ajax.php?action=obtener_detalle_venta&id=${ventaId}`)
            .then(res => res.json()).then(data => {
                if(data.success) {
                    renderizarDetalleModal(data.data);
                    modalDetalle.show();
                } else { Swal.fire('Error', data.message, 'error'); }
            });
    };

    function renderizarDetalleModal(data) {
        const { venta, items, cliente } = data;
        const modalBody = document.getElementById('venta-detalle-body');
        const modalFooter = document.getElementById('venta-detalle-footer');

        const productosHtml = items.map(item => `
            <tr>
                <td>${item.producto_nombre} (SKU: ${item.sku || 'N/A'})</td>
                <td class="text-center">${item.cantidad}</td>
                <td class="text-end">$${parseFloat(item.precio_unitario).toFixed(2)}</td>
                <td class="text-end">$${parseFloat(item.subtotal).toFixed(2)}</td>
            </tr>`).join('');

        const comprobanteHtml = venta.comprobante_pago 
            ? `<a href="/uploads/comprobantes/${venta.comprobante_pago}" target="_blank" class="btn btn-outline-info"><i class="fa-solid fa-receipt"></i> Ver Comprobante</a>`
            : '<span>No adjuntado</span>';

        const shippingHtml = (venta.estado === 'Verificado' || venta.estado === 'En Despacho' || venta.estado === 'Completado') ? `
            <div id="shipping-info-container">
                <h5>Datos de Envío</h5>
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Empresa de Envío</label>
                        <select id="empresa_envio" class="form-select" ${venta.estado !== 'Verificado' ? 'disabled' : ''}>
                            <option value="">Seleccionar...</option>
                            <option value="MRW" ${venta.empresa_envio === 'MRW' ? 'selected' : ''}>MRW</option>
                            <option value="ZOOM" ${venta.empresa_envio === 'ZOOM' ? 'selected' : ''}>ZOOM</option>
                        </select>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Número de Guía / Tracking</label>
                        <input type="text" id="numero_guia" class="form-control" value="${venta.numero_guia || ''}" ${venta.estado !== 'Verificado' ? 'disabled' : ''}>
                    </div>
                </div>
            </div>
        ` : '';

        modalBody.innerHTML = `
            <div class="row">
                <div class="col-lg-7">
                    <h4>Productos del Pedido</h4>
                    <div class="table-responsive">
                        <table class="ami-table">
                            <thead><tr><th>Producto</th><th class="text-center">Cant.</th><th class="text-end">Precio U.</th><th class="text-end">Subtotal</th></tr></thead>
                            <tbody>${productosHtml}</tbody>
                        </table>
                    </div>
                </div>
                <div class="col-lg-5">
                    <h4>Información General</h4>
                    <p><strong>Cliente:</strong> ${cliente.nombre}</p>
                    <p><strong>Dirección de Entrega:</strong><br>${venta.direccion_entrega}, ${venta.ciudad}, ${venta.estado_ve}</p>
                    <hr>
                    <p><strong>Monto Total:</strong> <span class="fs-4 fw-bold text-primary">$${parseFloat(venta.monto_total).toFixed(2)}</span></p>
                    <p><strong>Método de Pago:</strong> ${venta.metodo_pago}</p>
                    <p><strong>Referencia:</strong> ${venta.referencia_pago || 'N/A'}</p>
                    <p><strong>Comprobante:</strong> ${comprobanteHtml}</p>
                    <hr>
                    <p><strong>Observaciones:</strong><br>${venta.observaciones || 'Ninguna.'}</p>
                </div>
            </div>
            ${shippingHtml}
        `;
        
        modalFooter.innerHTML = '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>';
        modalFooter.innerHTML += `<button class="btn btn-outline-primary" onclick="generarPDF(${venta.id})"><i class="fa-solid fa-print"></i> Generar Orden</button>`;
        
        if (venta.estado !== 'Cancelado') {
            modalFooter.innerHTML += `<button class="btn btn-danger" onclick="cancelarVenta(${venta.id}, '${venta.estado}')"><i class="fa-solid fa-ban"></i> Cancelar Venta</button>`;
        }
        
        if (venta.estado === 'Pendiente') {
            modalFooter.innerHTML += `<button class="btn btn-success" onclick="actualizarEstado(${venta.id}, 'Verificado')"><i class="fa-solid fa-check-double"></i> Verificar y Descontar</button>`;
        } else if (venta.estado === 'Verificado') {
            modalFooter.innerHTML += `<button class="btn btn-primary" onclick="actualizarEstado(${venta.id}, 'En Despacho')"><i class="fa-solid fa-truck"></i> Marcar "En Despacho"</button>`;
        } else if (venta.estado === 'En Despacho') {
            modalFooter.innerHTML += `<button class="btn btn-success" onclick="actualizarEstado(${venta.id}, 'Completado')"><i class="fa-solid fa-flag-checkered"></i> Marcar "Completado"</button>`;
        }
    }

    window.generarPDF = function(ventaId) {
        window.open(`/api/pdf_despacho_ajax.php?id=${ventaId}`, '_blank');
    }

    window.cancelarVenta = function(ventaId, estadoActual) {
        let textoConfirmacion = '¿Estás seguro de que quieres cancelar esta venta? Esta acción no se puede deshacer.';
        if (estadoActual !== 'Pendiente' && estadoActual !== 'Cancelado') {
            textoConfirmacion += ' El stock de los productos asociados será devuelto al inventario.';
        }

        Swal.fire({
            title: 'Confirmar Cancelación',
            text: textoConfirmacion,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            confirmButtonText: 'Sí, cancelar venta',
            cancelButtonText: 'No, mantener'
        }).then((result) => {
            if (result.isConfirmed) {
                const formData = new FormData();
                formData.append('action', 'cancelar_venta');
                formData.append('venta_id', ventaId);
                formData.append('csrf_token', csrfToken);

                fetch('/api/despachos_ajax.php', { method: 'POST', body: formData })
                .then(res => res.json()).then(data => {
                    if(data.success) {
                        Swal.fire('¡Cancelada!', data.message, 'success');
                        modalDetalle.hide();
                        cargarVentas();
                    } else { Swal.fire('Error', data.message, 'error'); }
                });
            }
        });
    };

    window.actualizarEstado = function(ventaId, nuevoEstado) {
        const titulos = { 'Verificado': '¿Verificar Venta?', 'En Despacho': '¿Marcar como "En Despacho"?', 'Completado': '¿Marcar como "Completado"?' };
        const textos = { 'Verificado': 'Esta acción descontará los productos del inventario y registrará el movimiento.', 'En Despacho': 'El pedido se marcará como enviado.', 'Completado': 'La venta se finalizará.' };
        Swal.fire({
            title: titulos[nuevoEstado], text: textos[nuevoEstado], icon: 'question',
            showCancelButton: true, confirmButtonText: 'Sí, continuar', cancelButtonText: 'No'
        }).then((result) => {
            if (result.isConfirmed) {
                const formData = new FormData();
                formData.append('action', 'actualizar_estado_venta');
                formData.append('venta_id', ventaId);
                formData.append('nuevo_estado', nuevoEstado);
                formData.append('csrf_token', csrfToken);
                
                if (nuevoEstado === 'En Despacho') {
                    formData.append('empresa_envio', document.getElementById('empresa_envio').value);
                    formData.append('numero_guia', document.getElementById('numero_guia').value);
                }

                Swal.fire({ title: 'Procesando...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
                fetch('/api/despachos_ajax.php', { method: 'POST', body: formData })
                .then(res => res.json()).then(data => {
                    if(data.success) {
                        Swal.fire('¡Éxito!', data.message, 'success');
                        modalDetalle.hide();
                        cargarVentas();
                    } else { Swal.fire('Error', data.message, 'error'); }
                });
            }
        });
    };

    // LÓGICA PARA EL MODAL DE REPORTES
    document.getElementById('btnAbrirModalReporte').addEventListener('click', function() {
        // Establecer fechas por defecto al día de hoy
        const hoy = new Date();
        fpInicio.setDate(hoy);
        fpFin.setDate(hoy);
        modalReporte.show();
    });

    document.getElementById('btnGenerarReportePdf').addEventListener('click', function() {
        const fechaInicio = fpInicio.selectedDates[0] ? fpInicio.formatDate(fpInicio.selectedDates[0], "Y-m-d") : '';
        const fechaFin = fpFin.selectedDates[0] ? fpFin.formatDate(fpFin.selectedDates[0], "Y-m-d") : '';

        if (!fechaInicio || !fechaFin) {
            Swal.fire('Atención', 'Debes seleccionar una fecha de inicio y una fecha de fin.', 'warning');
            return;
        }
        if (new Date(fechaInicio) > new Date(fechaFin)) {
            Swal.fire('Atención', 'La fecha de inicio no puede ser mayor que la fecha de fin.', 'warning');
            return;
        }

        const url = `/api/despachos_reportes_pdf_ajax.php?fecha_inicio=${fechaInicio}&fecha_fin=${fechaFin}`;
        window.open(url, '_blank');
        modalReporte.hide();
    });

    document.getElementById('searchInput').addEventListener('input', renderizarTabla);
    cargarVentas();
});
</script>

<?php include_once 'templates/footer.php'; ?>