<?php
require_once '../administracion/includes/seguridad.php';

// Verificar permisos
if (!puede('gestionar_clientes_admin')) {
    header('Location: ../login.php');
    exit();
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Importar Clientes - Sistema</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        .upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 40px;
            text-align: center;
            transition: all 0.3s ease;
            background: #f8f9fa;
            cursor: pointer;
        }
        .upload-area:hover {
            border-color: #0d6efd;
            background: #e7f3ff;
        }
        .upload-area.dragover {
            border-color: #0d6efd;
            background: #e7f3ff;
            transform: scale(1.02);
        }
        .sheet-card {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 15px;
            margin: 10px 0;
            transition: all 0.3s ease;
            cursor: pointer;
            background: #fff;
        }
        .sheet-card:hover {
            border-color: #0d6efd;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .sheet-card.selected {
            border-color: #0d6efd;
            background: #e7f3ff;
        }
        .sheet-preview {
            max-height: 200px;
            overflow-y: auto;
            font-size: 11px;
        }
        .multi-sheet-selector {
            display: none;
            margin-top: 20px;
        }
        .progress-container, .preview-container, .results-container {
            display: none;
            margin-top: 20px;
        }
        .file-info {
            background: #e9ecef;
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
            flex-wrap: wrap;
        }
        .step {
            display: flex;
            align-items: center;
            padding: 0 15px;
            margin: 5px 0;
        }
        .step-circle {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #dee2e6;
            color: #6c757d;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 10px;
            font-size: 14px;
            font-weight: bold;
        }
        .step.active .step-circle {
            background: #0d6efd;
            color: white;
        }
        .step.completed .step-circle {
            background: #198754;
            color: white;
        }
        .sheet-stats {
            display: flex;
            justify-content: space-around;
            background: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            margin-top: 10px;
        }
        .debug-panel {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin: 20px 0;
            font-family: monospace;
            font-size: 12px;
            max-height: 200px;
            overflow-y: auto;
        }
        .table-responsive {
            max-height: 400px;
            overflow-y: auto;
        }
        .data-cleaning-info {
            background: #e7f3ff;
            border: 1px solid #b3d7ff;
            border-radius: 8px;
            padding: 15px;
            margin: 15px 0;
        }
        .cleaned-data-preview {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin: 10px 0;
        }
        .duplicados-list {
            max-height: 300px;
            overflow-y: auto;
        }
        .error-details {
            max-height: 300px;
            overflow-y: auto;
            background: #fff3cd;
            border: 1px solid #f5c6cb;
            border-radius: 5px;
            padding: 15px;
            margin-top: 15px;
        }
        .omitted-details {
            max-height: 300px;
            overflow-y: auto;
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            border-radius: 5px;
            padding: 15px;
            margin-top: 15px;
        }
        @media (max-width: 768px) {
            .step {
                padding: 0 5px;
                font-size: 12px;
            }
            .step-circle {
                width: 25px;
                height: 25px;
                font-size: 12px;
                margin-right: 5px;
            }
            .sheet-card {
                margin: 5px 0;
                padding: 10px;
            }
        }
    </style>
</head>
<body>
    <script>
    const CSRF_TOKEN = '<?php echo $_SESSION['csrf_token']; ?>';
    </script>
    <div class="container-fluid py-4">
        <div class="row">
            <div class="col-12">
                <!-- Panel de Debug -->
                <div class="alert alert-info">
                    <h6><i class="fas fa-bug me-2"></i>Panel de Debug</h6>
                    <div class="debug-panel" id="debugLog">
                        Esperando eventos...
                    </div>
                </div>

                <div class="card shadow-sm">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-file-excel me-2"></i>
                            Importar Clientes desde Excel
                        </h4>
                        <small>Selecciona las hojas que deseas procesar</small>
                    </div>
                    <div class="card-body">
                        <!-- Indicador de pasos -->
                        <div class="step-indicator">
                            <div class="step active" id="step1">
                                <div class="step-circle">1</div>
                                <span>Seleccionar Archivo</span>
                            </div>
                            <div class="step" id="step2">
                                <div class="step-circle">2</div>
                                <span>Seleccionar Hojas</span>
                            </div>
                            <div class="step" id="step3">
                                <div class="step-circle">3</div>
                                <span>Vista Previa</span>
                            </div>
                            <div class="step" id="step4">
                                <div class="step-circle">4</div>
                                <span>Importar</span>
                            </div>
                            <div class="step" id="step5">
                                <div class="step-circle">5</div>
                                <span>Resultados</span>
                            </div>
                        </div>

                        <!-- Información sobre limpieza de datos -->
                        <div class="data-cleaning-info">
                            <h6><i class="fas fa-magic me-2"></i>Limpieza Automática de Datos</h6>
                            <div class="row">
                                <div class="col-md-4">
                                    <strong>Nombres:</strong>
                                    <ul class="small mb-0">
                                        <li>Elimina emojis y caracteres especiales</li>
                                        <li>Si está vacío: "Sin Nombre"</li>
                                        <li>Capitaliza correctamente</li>
                                    </ul>
                                </div>
                                <div class="col-md-4">
                                    <strong>Teléfonos:</strong>
                                    <ul class="small mb-0">
                                        <li><code>4144533455</code> → <code>584144533455</code></li>
                                        <li><code>414-434343</code> → <code>58414434343</code></li>
                                        <li><code>04125435425</code> → <code>584125435425</code></li>
                                    </ul>
                                </div>
                                <div class="col-md-4">
                                    <strong>Textos:</strong>
                                    <ul class="small mb-0">
                                        <li>Elimina emojis</li>
                                        <li>Limpia espacios múltiples</li>
                                        <li>Normaliza formato</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <!-- Área de carga de archivo -->
                        <div id="upload-section">
                            <div class="upload-area" id="uploadArea">
                                <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                                <h5>Arrastra tu archivo Excel aquí</h5>
                                <p class="text-muted">Después podrás seleccionar qué hojas procesar</p>
                                <input type="file" id="fileInput" class="d-none" accept=".xlsx,.xls">
                                <button type="button" class="btn btn-outline-primary" id="selectFileBtn">
                                    <i class="fas fa-folder-open me-2"></i>Seleccionar Archivo Excel
                                </button>
                                <small class="d-block mt-2 text-muted">Tamaño máximo: 50MB | Formatos: .xlsx, .xls</small>
                            </div>
                            
                            <div class="file-info" id="fileInfo" style="display: none;">
                                <div class="row">
                                    <div class="col-md-4">
                                        <strong>Archivo:</strong> <span id="fileName"></span>
                                    </div>
                                    <div class="col-md-2">
                                        <strong>Tamaño:</strong> <span id="fileSize"></span>
                                    </div>
                                    <div class="col-md-2">
                                        <strong>Hojas:</strong> <span id="sheetsCount">Analizando...</span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Estado:</strong> <span id="fileStatus" class="badge bg-info">Procesando...</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Selector mejorado de hojas múltiples -->
                        <div class="multi-sheet-selector" id="multiSheetSelector">
                            <div class="alert alert-info">
                                <div class="row align-items-center">
                                    <div class="col-md-8">
                                        <h5 class="mb-1"><i class="fas fa-layers me-2"></i>Archivo con Múltiples Hojas Detectado</h5>
                                        <p class="mb-0">Selecciona las hojas que deseas importar. Puedes elegir una o varias hojas.</p>
                                    </div>
                                    <div class="col-md-4 text-center">
                                        <div class="btn-group" role="group">
                                            <button type="button" class="btn btn-outline-primary btn-sm" id="selectAllSheets">
                                                <i class="fas fa-check-double me-1"></i>Todas
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary btn-sm" id="clearAllSheets">
                                                <i class="fas fa-times me-1"></i>Ninguna
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="row" id="sheetsContainer">
                                <!-- Las hojas se cargarán aquí dinámicamente -->
                            </div>

                            <div class="text-center mt-3">
                                <div class="alert alert-secondary d-inline-block">
                                    <strong>Hojas seleccionadas:</strong> <span id="selectedSheetsCount">0</span> | 
                                    <strong>Registros estimados:</strong> <span id="totalEstimatedRecords">0</span>
                                </div>
                            </div>

                            <div class="text-center">
                                <button type="button" class="btn btn-success btn-lg" id="processSelectedSheets" disabled>
                                    <i class="fas fa-eye me-2"></i>Vista Previa de Hojas Seleccionadas
                                </button>
                            </div>
                        </div>

                        <!-- Vista previa combinada -->
                        <div class="preview-container" id="previewContainer">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h5><i class="fas fa-table me-2"></i>Vista Previa de Datos Combinados</h5>
                                <div>
                                    <button type="button" class="btn btn-warning me-2" id="backToSheetSelection">
                                        <i class="fas fa-arrow-left me-2"></i>Cambiar Hojas
                                    </button>
                                    <button type="button" class="btn btn-success" id="importBtn">
                                        <i class="fas fa-download me-2"></i>Iniciar Importación
                                    </button>
                                </div>
                            </div>
                            
                            <div class="alert alert-info">
                                <strong><i class="fas fa-map me-2"></i>Mapeo de Columnas:</strong>
                                <div class="row mt-2">
                                    <div class="col-md-2">
                                        <label class="form-label small">NOMBRE</label>
                                        <select class="form-select form-select-sm" id="mapNombre">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">TELÉFONO *</label>
                                        <select class="form-select form-select-sm" id="mapTelefono">
                                            <option value="">Seleccionar</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">CIUDAD</label>
                                        <select class="form-select form-select-sm" id="mapCiudad">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">CAMPAÑA</label>
                                        <select class="form-select form-select-sm" id="mapOrigen">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">FECHA</label>
                                        <select class="form-select form-select-sm" id="mapFecha">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2 d-flex align-items-end">
                                        <button type="button" class="btn btn-sm btn-outline-secondary w-100" onclick="actualizarVistaPrevia()">
                                            <i class="fas fa-sync-alt me-1"></i>Actualizar
                                        </button>
                                    </div>
                                </div>
                                <small class="text-muted">* Campo obligatorio</small>
                            </div>

                            <div class="table-responsive">
                                <table class="table table-sm table-striped" id="previewTable">
                                    <thead class="table-dark">
                                        <tr id="previewHeaders"></tr>
                                    </thead>
                                    <tbody id="previewBody"></tbody>
                                </table>
                            </div>
                        </div>

                        <!-- Barra de progreso -->
                        <div class="progress-container" id="progressContainer">
                            <div class="row">
                                <div class="col-md-8">
                                    <h5><i class="fas fa-cogs me-2"></i>Importando y limpiando datos...</h5>
                                    <div class="progress mb-3" style="height: 30px;">
                                        <div class="progress-bar progress-bar-striped progress-bar-animated bg-success" 
                                             id="progressBar" role="progressbar" style="width: 0%">
                                            <span id="progressText" class="fw-bold">0%</span>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card text-center">
                                        <div class="card-body p-3">
                                            <div class="row">
                                                <div class="col-3">
                                                    <small class="text-muted">Procesados</small>
                                                    <div class="h5" id="processedCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-success">Exitosos</small>
                                                    <div class="h5 text-success" id="successCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-warning">Duplicados</small>
                                                    <div class="h5 text-warning" id="duplicateCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-danger">Errores</small>
                                                    <div class="h5 text-danger" id="errorCount">0</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div id="statusMessage" class="alert alert-info">
                                <i class="fas fa-spinner fa-spin me-2"></i>Preparando importación...
                            </div>
                        </div>

                        <!-- Resultados -->
                        <div class="results-container" id="resultsContainer">
                            <div class="alert alert-success">
                                <h5><i class="fas fa-check-circle me-2"></i>¡Importación Completada!</h5>
                                <div class="row">
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-primary" id="finalProcessed">0</div>
                                        <small>Total Procesados</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-success" id="finalSuccess">0</div>
                                        <small>Importados</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-warning" id="finalDuplicates">0</div>
                                        <small>Duplicados</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-danger" id="finalErrors">0</div>
                                        <small>Errores</small>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Detalles de duplicados -->
                            <div id="duplicatesSection" style="display: none;">
                                <h6><i class="fas fa-exclamation-triangle text-warning me-2"></i>Registros Duplicados Encontrados:</h6>
                                <div class="duplicados-list border rounded p-3 bg-light">
                                    <div id="duplicatesList"></div>
                                </div>
                            </div>

                            <!-- Detalles de errores -->
                            <div id="errorsSection" style="display: none;">
                                <h6><i class="fas fa-times-circle text-danger me-2"></i>Errores Encontrados:</h6>
                                <div class="error-details">
                                    <div id="errorsList"></div>
                                </div>
                            </div>

                            <!-- Detalles de omitidos -->
                            <div id="omittedSection" style="display: none;">
                                <h6><i class="fas fa-info-circle text-info me-2"></i>Registros Omitidos (No Venezolanos):</h6>
                                <div class="omitted-details">
                                    <div id="omittedList"></div>
                                </div>
                            </div>

                            <div class="text-center mt-4">
                                <button type="button" class="btn btn-primary btn-lg me-3" onclick="location.reload()">
                                    <i class="fas fa-redo me-2"></i>Nueva Importación
                                </button>
                                <a href="clientes.php" class="btn btn-success btn-lg">
                                    <i class="fas fa-users me-2"></i>Ver Clientes
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script>
        let currentFile = null;
        let workbook = null;
        let selectedSheets = new Set();
        let sheetDataCache = new Map();
        let combinedData = null;
        let debugLog = [];

        // Función de debug
        function addDebugLog(message, type = 'info') {
            const timestamp = new Date().toLocaleTimeString();
            const logEntry = `[${timestamp}] ${type.toUpperCase()}: ${message}`;
            debugLog.push(logEntry);
            
            const debugPanel = document.getElementById('debugLog');
            if (debugPanel) {
                debugPanel.innerHTML = debugLog.slice(-20).join('\n');
                debugPanel.scrollTop = debugPanel.scrollHeight;
            }
            console.log(logEntry);
        }

        // Inicialización
        document.addEventListener('DOMContentLoaded', function() {
            addDebugLog('DOM cargado correctamente');
            
            if (typeof XLSX !== 'undefined') {
                addDebugLog('Librería XLSX disponible');
            } else {
                addDebugLog('ERROR: XLSX no disponible', 'error');
                mostrarAlerta('danger', 'Error: No se pudo cargar la librería XLSX');
                return;
            }
            
            setupEventListeners();
        });

        function setupEventListeners() {
            addDebugLog('Configurando event listeners...');
            
            const fileInput = document.getElementById('fileInput');
            const selectFileBtn = document.getElementById('selectFileBtn');
            const uploadArea = document.getElementById('uploadArea');
            
            fileInput?.addEventListener('change', handleFileSelect);
            selectFileBtn?.addEventListener('click', () => fileInput.click());
            
            // Drag and drop
            if (uploadArea) {
                uploadArea.addEventListener('click', (e) => {
                    if (!e.target.closest('button')) fileInput.click();
                });
                
                uploadArea.addEventListener('dragover', (e) => {
                    e.preventDefault();
                    uploadArea.classList.add('dragover');
                });
                
                uploadArea.addEventListener('dragleave', () => {
                    uploadArea.classList.remove('dragover');
                });
                
                uploadArea.addEventListener('drop', (e) => {
                    e.preventDefault();
                    uploadArea.classList.remove('dragover');
                    const files = e.dataTransfer.files;
                    if (files.length > 0) handleFile(files[0]);
                });
            }

            // Botones de selección
            document.getElementById('selectAllSheets')?.addEventListener('click', selectAllSheets);
            document.getElementById('clearAllSheets')?.addEventListener('click', clearAllSheets);
            document.getElementById('processSelectedSheets')?.addEventListener('click', processSelectedSheets);
            document.getElementById('backToSheetSelection')?.addEventListener('click', backToSheetSelection);
            document.getElementById('importBtn')?.addEventListener('click', startImport);
        }

        function handleFileSelect(event) {
            const files = event.target.files;
            if (files.length > 0) handleFile(files[0]);
        }

        function handleFile(file) {
            addDebugLog(`Procesando archivo: ${file.name}`);
            
            // Validaciones
            const validExtensions = ['.xlsx', '.xls'];
            const fileExtension = file.name.toLowerCase().substring(file.name.lastIndexOf('.'));
            
            if (!validExtensions.includes(fileExtension)) {
                mostrarAlerta('danger', 'Por favor selecciona un archivo Excel válido (.xlsx o .xls)');
                return;
            }

            if (file.size > 50 * 1024 * 1024) {
                mostrarAlerta('danger', 'El archivo es demasiado grande. Máximo permitido: 50MB');
                return;
            }

            currentFile = file;
            
            // Mostrar info del archivo
            document.getElementById('fileName').textContent = file.name;
            document.getElementById('fileSize').textContent = formatFileSize(file.size);
            document.getElementById('fileStatus').textContent = 'Analizando...';
            document.getElementById('fileStatus').className = 'badge bg-warning';
            document.getElementById('fileInfo').style.display = 'block';
            
            // Leer archivo
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    workbook = XLSX.read(e.target.result, { type: 'binary' });
                    processWorkbook();
                } catch (error) {
                    addDebugLog(`Error al parsear archivo: ${error.message}`, 'error');
                    mostrarAlerta('danger', 'Error al leer el archivo: ' + error.message);
                    document.getElementById('fileStatus').textContent = 'Error';
                    document.getElementById('fileStatus').className = 'badge bg-danger';
                }
            };
            reader.readAsBinaryString(file);
            updateStep(2);
        }

        function processWorkbook() {
            const sheetNames = workbook.SheetNames;
            addDebugLog(`Hojas encontradas: ${sheetNames.length}`);
            
            document.getElementById('sheetsCount').textContent = sheetNames.length;
            document.getElementById('fileStatus').textContent = 'Procesado';
            document.getElementById('fileStatus').className = 'badge bg-success';
            
            // Resetear selecciones
            selectedSheets.clear();
            sheetDataCache.clear();
            
            // Mostrar selector de hojas
            displaySheetSelector(sheetNames);
            document.getElementById('multiSheetSelector').style.display = 'block';
            
            mostrarAlerta('success', `Archivo procesado. ${sheetNames.length} hojas detectadas.`);
        }

        function displaySheetSelector(sheetNames) {
            const container = document.getElementById('sheetsContainer');
            container.innerHTML = '';
            
            sheetNames.forEach((sheetName, index) => {
                const worksheet = workbook.Sheets[sheetName];
                const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });
                const totalRows = jsonData.length;
                const dataRows = Math.max(0, totalRows - 1); // Sin contar header
                const headers = jsonData[0] || [];
                
                // Cachear datos de la hoja
                sheetDataCache.set(sheetName, jsonData);
                
                const colSize = sheetNames.length <= 2 ? 'col-md-6' : sheetNames.length <= 3 ? 'col-md-4' : 'col-md-3';
                
                const sheetCard = document.createElement('div');
                sheetCard.className = colSize;
                sheetCard.innerHTML = `
                    <div class="sheet-card" data-sheet="${sheetName}" onclick="toggleSheetSelection('${sheetName.replace(/'/g, "\\'")}')">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="sheet_${index}">
                            <label class="form-check-label fw-bold" for="sheet_${index}">
                                ${sheetName}
                            </label>
                        </div>
                        
                        <div class="sheet-stats">
                            <div class="text-center">
                                <small class="text-muted">Filas de Datos</small>
                                <div class="fw-bold text-primary">${dataRows.toLocaleString()}</div>
                            </div>
                            <div class="text-center">
                                <small class="text-muted">Columnas</small>
                                <div class="fw-bold text-info">${headers.length}</div>
                            </div>
                        </div>
                        
                        <div class="sheet-preview mt-2">
                            <small class="text-muted">Vista previa headers:</small>
                            <div class="small">
                                ${headers.slice(0, 4).map(h => `<span class="badge bg-light text-dark me-1">${h || 'Sin nombre'}</span>`).join('')}
                                ${headers.length > 4 ? `<span class="text-muted">... y ${headers.length - 4} más</span>` : ''}
                            </div>
                        </div>
                        
                        ${dataRows > 0 ? `
                            <div class="mt-2">
                                <small class="text-muted">Ejemplo de datos:</small>
                                <div class="small font-monospace text-truncate">
                                    ${jsonData[1] ? jsonData[1].slice(0, 3).join(' | ') : 'Sin datos'}
                                </div>
                            </div>
                        ` : '<div class="text-warning mt-2"><small>Hoja sin datos</small></div>'}
                    </div>
                `;
                
                container.appendChild(sheetCard);
            });
            
            updateSheetSelectionCounter();
        }

        function toggleSheetSelection(sheetName) {
            const checkbox = document.querySelector(`[data-sheet="${sheetName}"] input[type="checkbox"]`);
            const card = document.querySelector(`[data-sheet="${sheetName}"]`);
            
            if (selectedSheets.has(sheetName)) {
                selectedSheets.delete(sheetName);
                checkbox.checked = false;
                card.classList.remove('selected');
            } else {
                selectedSheets.add(sheetName);
                checkbox.checked = true;
                card.classList.add('selected');
            }
            
            updateSheetSelectionCounter();
            addDebugLog(`Hoja ${sheetName} ${selectedSheets.has(sheetName) ? 'seleccionada' : 'deseleccionada'}`);
        }

        function selectAllSheets() {
            document.querySelectorAll('.sheet-card').forEach(card => {
                const sheetName = card.dataset.sheet;
                const checkbox = card.querySelector('input[type="checkbox"]');
                selectedSheets.add(sheetName);
                checkbox.checked = true;
                card.classList.add('selected');
            });
            updateSheetSelectionCounter();
            addDebugLog('Todas las hojas seleccionadas');
        }

        function clearAllSheets() {
            selectedSheets.clear();
            document.querySelectorAll('.sheet-card').forEach(card => {
                const checkbox = card.querySelector('input[type="checkbox"]');
                checkbox.checked = false;
                card.classList.remove('selected');
            });
            updateSheetSelectionCounter();
            addDebugLog('Todas las hojas deseleccionadas');
        }

        function updateSheetSelectionCounter() {
            const count = selectedSheets.size;
            let totalRecords = 0;
            
            selectedSheets.forEach(sheetName => {
                const data = sheetDataCache.get(sheetName);
                if (data) {
                    totalRecords += Math.max(0, data.length - 1);
                }
            });
            
            document.getElementById('selectedSheetsCount').textContent = count;
            document.getElementById('totalEstimatedRecords').textContent = totalRecords.toLocaleString();
            
            const processBtn = document.getElementById('processSelectedSheets');
            processBtn.disabled = count === 0;
            
            if (count > 0) {
                processBtn.innerHTML = `<i class="fas fa-eye me-2"></i>Vista Previa de ${count} Hoja${count > 1 ? 's' : ''} (${totalRecords.toLocaleString()} registros)`;
            } else {
                processBtn.innerHTML = '<i class="fas fa-eye me-2"></i>Vista Previa de Hojas Seleccionadas';
            }
        }

        function processSelectedSheets() {
            if (selectedSheets.size === 0) {
                mostrarAlerta('warning', 'Por favor selecciona al menos una hoja');
                return;
            }

            addDebugLog(`Procesando ${selectedSheets.size} hojas seleccionadas`);
            
            let combinedHeaders = [];
            let combinedRows = [];
            let sheetSummary = [];
            
            // Procesar cada hoja seleccionada
            selectedSheets.forEach(sheetName => {
                const data = sheetDataCache.get(sheetName);
                if (!data || data.length <= 1) {
                    addDebugLog(`Hoja ${sheetName} está vacía, omitiendo`, 'warning');
                    return;
                }
                
                const headers = data[0] || [];
                const rows = data.slice(1);
                
                // Si es la primera hoja, usar sus headers como base
                if (combinedHeaders.length === 0) {
                    combinedHeaders = [...headers];
                } else {
                    // Verificar compatibilidad de headers (opcional)
                    if (headers.length !== combinedHeaders.length) {
                        addDebugLog(`Hoja ${sheetName} tiene ${headers.length} columnas vs ${combinedHeaders.length} esperadas`, 'warning');
                    }
                }
                
                // Agregar filas con referencia a la hoja de origen
                const processedRows = rows.map(row => {
                    const newRow = [...row];
                    // Opcional: agregar columna con nombre de hoja origen
                    // newRow.push(sheetName);
                    return newRow;
                });
                
                combinedRows = combinedRows.concat(processedRows);
                sheetSummary.push({
                    name: sheetName,
                    rows: rows.length,
                    headers: headers.length
                });
            });
            
            combinedData = [combinedHeaders, ...combinedRows];
            
            addDebugLog(`Datos combinados: ${combinedHeaders.length} columnas, ${combinedRows.length} filas`);
            
            // Mostrar resumen
            mostrarAlerta('success', 
                `Hojas combinadas exitosamente: ${sheetSummary.map(s => `${s.name} (${s.rows} filas)`).join(', ')}`
            );
            
            // Configurar mapeo y mostrar preview
            setupColumnMapping(combinedHeaders);
            displayPreview(combinedHeaders, combinedRows.slice(0, 10));
            
            document.getElementById('multiSheetSelector').style.display = 'none';
            document.getElementById('previewContainer').style.display = 'block';
            updateStep(3);
        }

        function backToSheetSelection() {
            document.getElementById('previewContainer').style.display = 'none';
            document.getElementById('multiSheetSelector').style.display = 'block';
            updateStep(2);
        }

        function setupColumnMapping(headers) {
            addDebugLog(`Configurando mapeo de columnas con ${headers.length} headers`);
            
            const mappers = ['mapNombre', 'mapTelefono', 'mapCiudad', 'mapOrigen', 'mapFecha'];
            
            mappers.forEach(mapperId => {
                const select = document.getElementById(mapperId);
                if (select) {
                    select.innerHTML = mapperId === 'mapTelefono' ? 
                        '<option value="">Seleccionar</option>' : 
                        '<option value="">No mapear</option>';
                    
                    headers.forEach((header, index) => {
                        const option = document.createElement('option');
                        option.value = index;
                        option.textContent = header || `Columna ${index + 1}`;
                        select.appendChild(option);
                    });
                    
                    // Event listener para actualización automática
                    select.addEventListener('change', function() {
                        addDebugLog(`Mapeo cambiado: ${mapperId} = ${this.value}`);
                        setTimeout(() => {
                            mostrarEjemploLimpieza();
                            if (combinedData && combinedData.length > 1) {
                                const headers = combinedData[0];
                                const dataRows = combinedData.slice(1, 11);
                                displayPreview(headers, dataRows);
                            }
                        }, 100);
                    });
                }
            });
            
            autoMapColumns(headers);
            mostrarEjemploLimpieza();
        }

        function autoMapColumns(headers) {
            const mappings = {
                'mapNombre': ['nombre', 'name', 'cliente', 'razon social'],
                'mapTelefono': ['telefono', 'teléfono', 'phone', 'celular', 'tel'],
                'mapCiudad': ['ciudad', 'city', 'localidad'],
                'mapOrigen': ['origen', 'campaña', 'campaign', 'fuente'],
                'mapFecha': ['fecha', 'date', 'created', 'registro']
            };
            
            Object.keys(mappings).forEach(mapId => {
                const keywords = mappings[mapId];
                const headerIndex = headers.findIndex(header => {
                    if (!header) return false;
                    return keywords.some(keyword => 
                        header.toLowerCase().includes(keyword.toLowerCase())
                    );
                });
                
                if (headerIndex !== -1) {
                    const selectElement = document.getElementById(mapId);
                    if (selectElement) {
                        selectElement.value = headerIndex;
                    }
                }
            });
        }

        function mostrarEjemploLimpieza() {
            addDebugLog('Mostrando ejemplo de limpieza...');
            
            if (combinedData && combinedData.length > 1) {
                const ejemploData = combinedData[1];
                const mapping = {
                    nombre: document.getElementById('mapNombre')?.value || '',
                    telefono: document.getElementById('mapTelefono')?.value || ''
                };
                
                if (ejemploData && (mapping.nombre !== '' || mapping.telefono !== '')) {
                    const before = [];
                    const after = [];
                    
                    if (mapping.nombre !== '' && ejemploData[mapping.nombre] !== undefined) {
                        const valorOriginal = ejemploData[mapping.nombre] || '';
                        const valorLimpio = limpiarNombre(valorOriginal);
                        before.push(`Nombre: "${valorOriginal}"`);
                        after.push(`Nombre: "${valorLimpio}"`);
                    }
                    
                    if (mapping.telefono !== '' && ejemploData[mapping.telefono] !== undefined) {
                        const valorOriginal = ejemploData[mapping.telefono] || '';
                        const valorLimpio = limpiarTelefono(valorOriginal);
                        before.push(`Teléfono: "${valorOriginal}"`);
                        after.push(`Teléfono: "${valorLimpio}"`);
                    }
                    
                    // Mostrar el ejemplo si hay cambios
                    if (before.length > 0) {
                        // Crear el panel de ejemplo si no existe
                        let cleanedPreview = document.getElementById('cleanedDataPreview');
                        if (!cleanedPreview) {
                            const previewContainer = document.getElementById('previewContainer');
                            const mappingAlert = previewContainer.querySelector('.alert');
                            
                            cleanedPreview = document.createElement('div');
                            cleanedPreview.id = 'cleanedDataPreview';
                            cleanedPreview.className = 'alert alert-light border';
                            cleanedPreview.innerHTML = `
                                <h6><i class="fas fa-sparkles me-2"></i>Ejemplo de datos después de la limpieza:</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <strong>Antes:</strong>
                                        <div class="small font-monospace text-muted" id="dataBefore"></div>
                                    </div>
                                    <div class="col-md-6">
                                        <strong>Después:</strong>
                                        <div class="small font-monospace text-success" id="dataAfter"></div>
                                    </div>
                                </div>
                            `;
                            
                            mappingAlert.parentNode.insertBefore(cleanedPreview, mappingAlert.nextSibling);
                        }
                        
                        const beforeElement = document.getElementById('dataBefore');
                        const afterElement = document.getElementById('dataAfter');
                        
                        beforeElement.innerHTML = before.join('<br>');
                        afterElement.innerHTML = after.join('<br>');
                        cleanedPreview.style.display = 'block';
                    }
                }
            }
        }

        function actualizarVistaPrevia() {
            addDebugLog('Actualizando vista previa manualmente...');
            mostrarEjemploLimpieza();
            
            if (combinedData && combinedData.length > 1) {
                const headers = combinedData[0];
                const dataRows = combinedData.slice(1, 11);
                displayPreview(headers, dataRows);
            }
        }

        function displayPreview(headers, dataRows) {
            addDebugLog(`Mostrando preview con ${headers.length} headers y ${dataRows.length} filas`);
            
            const previewHeaders = document.getElementById('previewHeaders');
            const previewBody = document.getElementById('previewBody');
            
            if (!previewHeaders || !previewBody) {
                addDebugLog('ERROR: Elementos de preview no encontrados', 'error');
                return;
            }
            
            // Headers
            previewHeaders.innerHTML = headers.map(header => 
                `<th>${header || 'Sin nombre'}</th>`
            ).join('');
            
            // Obtener mapping actual
            const mapping = {
                nombre: document.getElementById('mapNombre')?.value || '',
                telefono: document.getElementById('mapTelefono')?.value || ''
            };
            
            // Rows con limpieza visual
            previewBody.innerHTML = dataRows.map((row) => {
                if (!row) return '';
                
                return `<tr>${headers.map((_, index) => {
                    let valor = row[index] !== undefined ? String(row[index]) : '';
                    let claseExtra = '';
                    let valorMostrar = valor;
                    
                    // Aplicar limpieza visual y colores
                    if (index == mapping.nombre && mapping.nombre !== '' && valor) {
                        valorMostrar = limpiarNombre(valor);
                        claseExtra = 'table-success';
                    } else if (index == mapping.telefono && mapping.telefono !== '' && valor) {
                        valorMostrar = limpiarTelefono(valor);
                        claseExtra = 'table-info';
                    }
                    
                    // Limitar longitud para mostrar
                    if (valorMostrar.length > 30) {
                        valorMostrar = valorMostrar.substring(0, 27) + '...';
                    }
                    
                    return `<td class="${claseExtra}" title="${valor}">${valorMostrar}</td>`;
                }).join('')}</tr>`;
            }).join('');
        }

        function startImport() {
            addDebugLog('Iniciando importación...');
            
            const mapping = {
                nombre: document.getElementById('mapNombre')?.value || '',
                telefono: document.getElementById('mapTelefono')?.value || '',
                ciudad: document.getElementById('mapCiudad')?.value || '',
                origen: document.getElementById('mapOrigen')?.value || '',
                fecha: document.getElementById('mapFecha')?.value || ''
            };
            
            if (!mapping.telefono) {
                mostrarAlerta('danger', 'El campo TELÉFONO es obligatorio para mapear');
                return;
            }
            
            if (!combinedData || combinedData.length <= 1) {
                mostrarAlerta('danger', 'No hay datos para procesar');
                return;
            }
            
            const dataRows = combinedData.slice(1); // Excluir headers
            
            // Validación previa rápida
            const validacionRapida = validarDatosRapido(dataRows, mapping);
            addDebugLog(`Validación previa: ${validacionRapida.validos}/${validacionRapida.total} registros válidos`);
            
            if (validacionRapida.validos === 0) {
                mostrarAlerta('danger', 'No se encontraron registros válidos para importar. Verifica el mapeo de columnas.');
                return;
            }
            
            if (validacionRapida.validos < validacionRapida.total * 0.5) {
                if (!confirm(`Solo ${validacionRapida.validos} de ${validacionRapida.total} registros parecen válidos (${Math.round(validacionRapida.validos/validacionRapida.total*100)}%). ¿Deseas continuar?`)) {
                    return;
                }
            }
            
            document.getElementById('previewContainer').style.display = 'none';
            document.getElementById('progressContainer').style.display = 'block';
            updateStep(4);
            
            // Resetear contadores
            document.getElementById('errorCount').textContent = '0';
            document.getElementById('duplicateCount').textContent = '0';
            document.getElementById('successCount').textContent = '0';
            document.getElementById('processedCount').textContent = '0';
            
            // Inicializar estadísticas
            const initialStats = {
                processed: 0,
                success: 0,
                duplicates: 0,
                errors: 0,
                duplicatesList: [],
                omittedList: [],
                errorsList: []
            };
            
            processBatch(dataRows, mapping, 0, initialStats);
        }

        function validarDatosRapido(dataRows, mapping) {
            let validos = 0;
            const total = dataRows.length;
            
            dataRows.forEach(row => {
                if (mapping.telefono !== '' && row[mapping.telefono]) {
                    const telefono = String(row[mapping.telefono]).trim();
                    if (telefono && telefono.length >= 7) {
                        validos++;
                    }
                }
            });
            
            return { validos, total };
        }

        function processBatch(allRows, mapping, startIndex, stats) {
            const batchSize = 50;
            const endIndex = Math.min(startIndex + batchSize, allRows.length);
            const batch = allRows.slice(startIndex, endIndex);
            
            if (batch.length === 0) {
                showResults(stats);
                return;
            }
            
            // Preparar datos del lote con limpieza
            const batchData = batch.map((row, index) => {
                const cliente = {
                    fila_original: startIndex + index + 2 // +2 para compensar header y empezar en 1
                };
                
                Object.keys(mapping).forEach(field => {
                    const colIndex = mapping[field];
                    if (colIndex !== '' && colIndex !== null && colIndex !== undefined) {
                        let valor = row[colIndex] || '';
                        
                        if (field === 'nombre') {
                            valor = limpiarNombre(valor);
                        } else if (field === 'telefono') {
                            valor = limpiarTelefono(valor);
                        } else if (field === 'ciudad') {
                            valor = limpiarTexto(valor);
                        }
                        
                        cliente[field] = valor;
                    }
                });
                return cliente;
            }).filter(cliente => cliente.telefono && cliente.telefono.length >= 10);
            
            addDebugLog(`Enviando lote ${Math.floor(startIndex/batchSize) + 1}: ${batchData.length} registros válidos de ${batch.length} procesados`);
            
            // Actualizar mensaje de estado
            document.getElementById('statusMessage').innerHTML = 
                `<i class="fas fa-cogs fa-spin me-2"></i>Procesando lote ${Math.floor(startIndex/batchSize) + 1}... Limpiando y validando ${batchData.length} registros`;
            
            // Enviar lote vía AJAX
            const formData = new FormData();
            formData.append('action', 'importar_clientes_lote');
            formData.append('clientes', JSON.stringify(batchData));
            formData.append('csrf_token', CSRF_TOKEN);
            
            fetch('ajax_clientes.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                addDebugLog(`Respuesta del servidor para lote ${Math.floor(startIndex/batchSize) + 1}: ${JSON.stringify(data.stats || {})}`);
                
                if (data.success && data.stats) {
                    // Acumular estadísticas
                    stats.processed += data.stats.processed || 0;
                    stats.success += data.stats.success || 0;
                    stats.duplicates += data.stats.duplicates || 0;
                    stats.errors += data.stats.errors || 0;
                    
                    // Acumular listas de detalles
                    if (data.stats.duplicatesList) {
                        stats.duplicatesList.push(...data.stats.duplicatesList);
                    }
                    if (data.stats.omittedList) {
                        if (!stats.omittedList) stats.omittedList = [];
                        stats.omittedList.push(...data.stats.omittedList);
                    }
                    if (data.stats.errorsList) {
                        if (!stats.errorsList) stats.errorsList = [];
                        stats.errorsList.push(...data.stats.errorsList);
                    }
                    
                    const progress = Math.round((endIndex / allRows.length) * 100);
                    updateProgress(progress, stats);
                    
                    // Continuar con el siguiente lote
                    if (endIndex < allRows.length) {
                        setTimeout(() => {
                            processBatch(allRows, mapping, endIndex, stats);
                        }, 200);
                    } else {
                        // Importación completada
                        document.getElementById('statusMessage').innerHTML = 
                            `<i class="fas fa-check text-success me-2"></i>Importación completada. Total procesado: ${stats.processed.toLocaleString()}`;
                        document.getElementById('statusMessage').className = 'alert alert-success';
                        
                        setTimeout(() => showResults(stats), 1000);
                    }
                } else {
                    addDebugLog(`Error del servidor: ${data.message}`, 'error');
                    throw new Error(data.message || 'Error desconocido del servidor');
                }
            })
            .catch(error => {
                addDebugLog(`Error en el lote ${Math.floor(startIndex/batchSize) + 1}: ${error.message}`, 'error');
                mostrarAlerta('danger', `Error en la importación: ${error.message}`);
                
                // Agregar errores del lote fallido
                stats.errors += batchData.length;
                updateProgress(Math.round((endIndex / allRows.length) * 100), stats);
                
                // Decidir si continuar o parar
                if (endIndex < allRows.length) {
                    // Continuar con el siguiente lote después del error
                    setTimeout(() => {
                        processBatch(allRows, mapping, endIndex, stats);
                    }, 1000);
                } else {
                    showResults(stats);
                }
            });
        }

        // Funciones de limpieza
        function limpiarNombre(nombre) {
            if (!nombre || typeof nombre !== 'string' || nombre.trim() === '') {
                return 'Sin Nombre';
            }
            
            nombre = nombre.trim();
            nombre = nombre.replace(/\w\S*/g, (txt) => 
                txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase()
            );
            
            return nombre || 'Sin Nombre';
        }

        function limpiarTelefono(telefono) {
            if (!telefono || typeof telefono !== 'string') {
                return '';
            }
            
            telefono = telefono.replace(/[^0-9]/g, '');
            
            if (!telefono) return '';
            
            if (telefono.startsWith('58') && telefono.length > 10) {
                telefono = telefono.substring(2);
            }
            
            if (telefono.startsWith('0')) {
                telefono = telefono.substring(1);
            }
            
            if (telefono.length === 10 && telefono.startsWith('4')) {
                return '58' + telefono;
            }
            
            if (telefono.length === 7) {
                return '58414' + telefono;
            }
            
            return '58' + telefono;
        }

        function limpiarTexto(texto) {
            if (!texto || typeof texto !== 'string') {
                return '';
            }
            return texto.trim();
        }

        function updateProgress(percentage, stats) {
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            progressBar.style.width = percentage + '%';
            progressText.textContent = percentage + '%';
            
            document.getElementById('processedCount').textContent = stats.processed.toLocaleString();
            document.getElementById('successCount').textContent = stats.success.toLocaleString();
            document.getElementById('duplicateCount').textContent = stats.duplicates.toLocaleString();
            document.getElementById('errorCount').textContent = stats.errors.toLocaleString();
            
            if (percentage === 100) {
                document.getElementById('statusMessage').innerHTML = 
                    `<i class="fas fa-check text-success me-2"></i>Importación completada`;
                document.getElementById('statusMessage').className = 'alert alert-success';
            }
        }

        function showResults(stats) {
            document.getElementById('progressContainer').style.display = 'none';
            document.getElementById('resultsContainer').style.display = 'block';
            
            document.getElementById('finalProcessed').textContent = stats.processed.toLocaleString();
            document.getElementById('finalSuccess').textContent = stats.success.toLocaleString();
            document.getElementById('finalDuplicates').textContent = stats.duplicates.toLocaleString();
            document.getElementById('finalErrors').textContent = stats.errors.toLocaleString();
            
            // Mostrar duplicados si existen
            if (stats.duplicatesList && stats.duplicatesList.length > 0) {
                const duplicatesSection = document.getElementById('duplicatesSection');
                document.getElementById('duplicatesList').innerHTML = stats.duplicatesList.map(dup => 
                    `<div class="mb-2 p-2 bg-white rounded border">
                        <strong>${dup.nombre}</strong> - ${dup.telefono}
                        <small class="text-muted d-block">Fila ${dup.fila || 'N/A'} - Ya existe en la base de datos</small>
                    </div>`
                ).join('');
                duplicatesSection.style.display = 'block';
            }
            
            // Mostrar errores si existen
            if (stats.errorsList && stats.errorsList.length > 0) {
                const errorsSection = document.getElementById('errorsSection');
                document.getElementById('errorsList').innerHTML = stats.errorsList.map(err => 
                    `<div class="mb-2 p-2 bg-white rounded border">
                        <strong>Fila ${err.fila}</strong> - ${err.nombre || 'Sin nombre'} - ${err.telefono || 'Sin teléfono'}
                        <small class="text-danger d-block">${err.error}</small>
                    </div>`
                ).join('');
                errorsSection.style.display = 'block';
            }
            
            // Mostrar omitidos si existen
            if (stats.omittedList && stats.omittedList.length > 0) {
                const omittedSection = document.getElementById('omittedSection');
                document.getElementById('omittedList').innerHTML = stats.omittedList.map(omit => 
                    `<div class="mb-2 p-2 bg-white rounded border">
                        <strong>${omit.nombre}</strong> - ${omit.telefono}
                        <small class="text-info d-block">Fila ${omit.fila || 'N/A'} - Teléfono no venezolano</small>
                    </div>`
                ).join('');
                omittedSection.style.display = 'block';
            }
            
            updateStep(5);
            addDebugLog('Importación completada exitosamente');
        }

        function updateStep(stepNumber) {
            document.querySelectorAll('.step').forEach(step => {
                step.classList.remove('active');
            });
            
            for (let i = 1; i < stepNumber; i++) {
                const step = document.getElementById(`step${i}`);
                if (step) {
                    step.classList.add('completed');
                    step.classList.remove('active');
                }
            }
            
            if (stepNumber <= 5) {
                const currentStep = document.getElementById(`step${stepNumber}`);
                if (currentStep) {
                    currentStep.classList.add('active');
                }
            }
        }

        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        function mostrarAlerta(tipo, mensaje) {
            const alertContainer = document.createElement('div');
            alertContainer.className = `alert alert-${tipo} alert-dismissible fade show position-fixed`;
            alertContainer.style.cssText = 'top: 20px; right: 20px; z-index: 1055; min-width: 300px;';
            alertContainer.innerHTML = `
                ${mensaje}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(alertContainer);
            
            setTimeout(() => {
                if (alertContainer.parentNode) {
                    alertContainer.remove();
                }
            }, 5000);
        }

        addDebugLog('Script cargado completamente');
    </script>
</body>
</html>