<?php
require_once '../includes/seguridad.php'; 
$page_title = 'Gestión de Equipos';
include_once '../templates/header.php';

if (!puede('gestionar_equipos')) {
    echo "<main class='ami-content'><div class='info-card'><h1>Acceso Denegado</h1></div></main>";
    include_once '../templates/footer.php';
    exit();
}
?>

<style>
.team-management-container {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 2rem;
}
.supervisor-list .list-group-item {
    cursor: pointer;
    background-color: rgba(255, 255, 255, 0.05);
    border-color: var(--border-color);
    color: var(--light-text);
}
.supervisor-list .list-group-item.active {
    background-color: var(--primary-color);
    color: var(--dark-text);
    border-color: var(--primary-color);
    font-weight: bold;
}
.operators-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
}
.operator-card {
    background-color: rgba(0,0,0,0.2);
    padding: 1rem;
    border-radius: 8px;
}
.operator-card h6 {
    color: var(--primary-color);
    margin-bottom: 1rem;
    border-bottom: 1px solid var(--border-color);
    padding-bottom: 0.5rem;
}
.operator-list {
    max-height: 400px;
    overflow-y: auto;
    padding-right: 10px;
}
.operator-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background-color: rgba(255,255,255,0.05);
    padding: 0.5rem 1rem;
    border-radius: 5px;
    margin-bottom: 0.5rem;
}
.btn-transfer {
    background: transparent;
    border: 1px solid var(--border-color);
    color: var(--light-text);
    width: 30px;
    height: 30px;
    border-radius: 50%;
    cursor: pointer;
}
</style>

<div class="main-header">
    <h1><?php echo $page_title; ?></h1>
</div>

<div class="info-card">
    <p>Selecciona un supervisor para ver y modificar su equipo de asesores.</p>
    <div class="team-management-container mt-4">
        <div>
            <h5>Supervisores</h5>
            <div class="list-group supervisor-list" id="supervisor-list-container">
                </div>
        </div>

        <div id="team-editor" class="d-none">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h4 id="supervisor-name-title"></h4>
                <button class="btn-add-new" id="btn-save-team"><i class="fa-solid fa-save"></i> Guardar Cambios</button>
            </div>
            <div class="operators-grid">
                <div class="operator-card">
                    <h6><i class="fa-solid fa-users"></i> Asesores en el Equipo</h6>
                    <div class="operator-list" id="team-operators-list"></div>
                </div>
                <div class="operator-card">
                    <h6><i class="fa-solid fa-user-plus"></i> Asesores Disponibles</h6>
                    <div class="operator-list" id="available-operators-list"></div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let supervisores = [];
    let operadores = [];
    let selectedSupervisorId = null;
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';

    async function cargarDatos() {
        try {
            const response = await fetch('/api/equipos_ajax.php?action=cargar_datos_gestion_equipos');
            const data = await response.json();
            if (data.success) {
                supervisores = data.data.supervisores;
                operadores = data.data.operadores;
                renderizarListaSupervisores();
            } else {
                Swal.fire('Error', data.message, 'error');
            }
        } catch (error) {
            Swal.fire('Error de Conexión', 'No se pudieron cargar los datos.', 'error');
        }
    }

    function renderizarListaSupervisores() {
        const container = document.getElementById('supervisor-list-container');
        container.innerHTML = supervisores.map(s => `
            <a href="#" class="list-group-item list-group-item-action" data-supervisor-id="${s.id}">
                ${s.nombre}
            </a>
        `).join('');

        container.querySelectorAll('.list-group-item').forEach(item => {
            item.addEventListener('click', function(e) {
                e.preventDefault();
                container.querySelector('.active')?.classList.remove('active');
                this.classList.add('active');
                selectedSupervisorId = parseInt(this.dataset.supervisorId);
                renderizarEditorDeEquipo(selectedSupervisorId);
            });
        });
    }

    function renderizarEditorDeEquipo(supervisorId) {
        const supervisor = supervisores.find(s => s.id === supervisorId);
        if (!supervisor) return;

        document.getElementById('supervisor-name-title').textContent = `Equipo de ${supervisor.nombre}`;
        
        const teamList = document.getElementById('team-operators-list');
        const availableList = document.getElementById('available-operators-list');
        teamList.innerHTML = '';
        availableList.innerHTML = '';

        operadores.forEach(op => {
            const itemHtml = `
                <div class="operator-item" data-operator-id="${op.id}">
                    <span>${op.nombre}</span>
                    <button class="btn-transfer">${op.supervisor_id === supervisorId ? '<i class="fa-solid fa-arrow-right"></i>' : '<i class="fa-solid fa-arrow-left"></i>'}</button>
                </div>
            `;
            if (op.supervisor_id === supervisorId) {
                teamList.innerHTML += itemHtml;
            } else if (!op.supervisor_id) {
                availableList.innerHTML += itemHtml;
            }
        });

        document.getElementById('team-editor').classList.remove('d-none');
        addTransferListeners();
    }

    function addTransferListeners() {
        document.querySelectorAll('.btn-transfer').forEach(button => {
            button.addEventListener('click', function() {
                const operatorItem = this.closest('.operator-item');
                const operatorId = parseInt(operatorItem.dataset.operatorId);
                const targetListId = this.closest('.operator-list').id === 'team-operators-list' ? 'available-operators-list' : 'team-operators-list';
                
                document.getElementById(targetListId).appendChild(operatorItem);
                this.innerHTML = targetListId === 'available-operators-list' ? '<i class="fa-solid fa-arrow-left"></i>' : '<i class="fa-solid fa-arrow-right"></i>';

                // Actualizar el estado en el array de operadores
                const op = operadores.find(o => o.id === operatorId);
                if (op) {
                    op.supervisor_id = targetListId === 'team-operators-list' ? selectedSupervisorId : null;
                }
            });
        });
    }

    document.getElementById('btn-save-team').addEventListener('click', async function() {
        if (!selectedSupervisorId) return;

        const teamOperatorIds = Array.from(document.getElementById('team-operators-list').querySelectorAll('.operator-item'))
            .map(item => item.dataset.operatorId);

        Swal.fire({ title: 'Guardando...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
        
        const formData = new FormData();
        formData.append('action', 'guardar_equipo_supervisor');
        formData.append('supervisor_id', selectedSupervisorId);
        teamOperatorIds.forEach(id => formData.append('operadores_ids[]', id));
        formData.append('csrf_token', csrfToken);
        
        try {
            const response = await fetch('/api/equipos_ajax.php', { method: 'POST', body: formData });
            const data = await response.json();

            if (data.success) {
                Swal.fire('¡Éxito!', data.message, 'success');
                // Recargar datos para reflejar los cambios en otros supervisores si es necesario
                cargarDatos(); 
                document.getElementById('team-editor').classList.add('d-none');
            } else {
                Swal.fire('Error', data.message, 'error');
            }
        } catch (error) {
            Swal.fire('Error de Conexión', 'No se pudieron guardar los cambios.', 'error');
        }
    });

    cargarDatos();
});
</script>


<?php include_once '../templates/footer.php'; ?>