<?php
define('IS_AJAX', true);
// NOTA: Asegúrate de que la ruta a seguridad.php sea correcta.
require_once __DIR__ . '/../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

// Validación CSRF
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? '';

if (!puede('gestionar_inventario_admin')) {
    $response['message'] = 'Acceso denegado.';
    echo json_encode($response);
    exit();
}

// --- FUNCIONES DE AYUDA (CORREGIDAS) ---

/**
 * Busca un producto por SKU. Si no existe, lo crea usando los datos del Excel.
 * Devuelve el ID del producto.
 */
function obtener_o_crear_producto_id($conn, $datos_producto, $usuario_id) {
    $sku = limpiar_texto($datos_producto['sku'] ?? null);
    if (empty($sku)) return null;

    // 1. Buscar por SKU, que debe ser único.
    $stmt = $conn->prepare("SELECT id FROM almacen_productos WHERE sku = ?");
    $stmt->execute([$sku]);
    $id = $stmt->fetchColumn();

    if ($id) {
        return $id; // Devuelve el ID si ya existe
    } else {
        // 2. Si no existe, lo crea.
        $sql_insert = "INSERT INTO almacen_productos (nombre, sku, descripcion, categoria, stock_minimo, creado_por) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt_insert = $conn->prepare($sql_insert);
        
        $params = [
            limpiar_texto($datos_producto['nombre'] ?? 'Sin Nombre'),
            $sku,
            limpiar_texto($datos_producto['descripcion'] ?? null),
            limpiar_texto($datos_producto['categoria'] ?? 'General'),
            limpiar_entero($datos_producto['stock_minimo'] ?? 0),
            $usuario_id
        ];
        
        $stmt_insert->execute($params);
        return $conn->lastInsertId();
    }
}

// --- FUNCIONES DE LIMPIEZA (Sin cambios) ---
function limpiar_entero($valor) { return !empty($valor) ? intval(preg_replace('/[^0-9]/', '', $valor)) : 0; }
function limpiar_texto($texto) { return !empty($texto) ? trim(preg_replace('/\s+/', ' ', $texto)) : null; }

// --- LÓGICA PRINCIPAL (CORREGIDA) ---

switch ($action) {
    case 'importar_inventario_lote':
        try {
            $productos_importados = json_decode($_POST['productos'] ?? '[]', true);
            if (!is_array($productos_importados)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }

            $stats = ['processed' => 0, 'success' => 0, 'errors' => 0, 'errorsList' => []];
            
            $conn->beginTransaction();

            foreach ($productos_importados as $producto_data) {
                $stats['processed']++;

                $sku = limpiar_texto($producto_data['sku'] ?? '');
                $nombre = limpiar_texto($producto_data['nombre'] ?? '');
                $cantidad_ingreso = limpiar_entero($producto_data['cantidad'] ?? 0);

                // Validación básica
                if (empty($sku) || empty($nombre) || $cantidad_ingreso <= 0) {
                    $stats['errors']++;
                    $stats['errorsList'][] = ['fila' => $producto_data['fila_original'], 'error' => 'SKU, Nombre y Cantidad (mayor a 0) son obligatorios.'];
                    continue;
                }

                // 1. Obtener o crear el producto en el catálogo
                $producto_id = obtener_o_crear_producto_id($conn, $producto_data, $usuario_id);
                if (!$producto_id) {
                    $stats['errors']++;
                    $stats['errorsList'][] = ['fila' => $producto_data['fila_original'], 'error' => 'No se pudo obtener o crear el producto en el catálogo.'];
                    continue;
                }

                // 2. Actualizar o insertar en la tabla de stock (Lógica UPSERT)
                $stmt_stock_check = $conn->prepare("SELECT cantidad FROM almacen_stock WHERE producto_id = ?");
                $stmt_stock_check->execute([$producto_id]);
                $cantidad_actual = $stmt_stock_check->fetchColumn();

                if ($cantidad_actual !== false) { // El producto ya tiene stock, se actualiza
                    $stmt_update_stock = $conn->prepare("UPDATE almacen_stock SET cantidad = cantidad + ? WHERE producto_id = ?");
                    $stmt_update_stock->execute([$cantidad_ingreso, $producto_id]);
                } else { // Es un producto nuevo en stock, se inserta
                    $stmt_insert_stock = $conn->prepare("INSERT INTO almacen_stock (producto_id, cantidad, ubicacion) VALUES (?, ?, ?)");
                    $stmt_insert_stock->execute([$producto_id, $cantidad_ingreso, 'Almacén Principal']);
                }

                // 3. Registrar el movimiento de ingreso
                $sql_movimiento = "INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, usuario_id) VALUES (?, ?, ?, ?, ?)";
                $stmt_movimiento = $conn->prepare($sql_movimiento);
                $stmt_movimiento->execute([
                    $producto_id,
                    'ingreso',
                    $cantidad_ingreso,
                    'Carga inicial desde Excel',
                    $usuario_id
                ]);
                
                $stats['success']++;
            }

            $conn->commit();
            // Eliminamos la sección de duplicados porque la nueva lógica los suma.
            $response = ['success' => true, 'message' => 'Lote procesado.', 'stats' => $stats];

        } catch (Exception $e) {
            if ($conn->inTransaction()) {
                $conn->rollBack();
            }
            $response['message'] = 'Error al procesar lote: ' . $e->getMessage();
            error_log("Error en importar_inventario_lote: " . $e->getMessage());
        }
        break;

    default:
        $response['message'] = 'Acción desconocida: ' . $action;
        break;
}

echo json_encode($response);