<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

if (!puede('gestionar_clientes_admin')) {
    $response['message'] = 'Acceso denegado.';
    echo json_encode($response);
    exit();
}

/**
 * Verifica si un número tiene el formato de un celular venezolano.
 */
function es_telefono_venezolano($telefono) {
    $numeros = preg_replace('/[^0-9]/', '', $telefono);
    $len = strlen($numeros);
    if ($len === 12 && strpos($numeros, '584') === 0) return true;
    if ($len === 11 && strpos($numeros, '04') === 0) return true;
    if ($len === 10 && strpos($numeros, '4') === 0) return true;
    return false;
}

/**
 * Normaliza un número de teléfono venezolano al formato estándar 584...
 */
function normalizar_telefono_venezolano($telefono) {
    if (empty($telefono)) return '';
    $numeros = preg_replace('/[^0-9]/', '', $telefono);
    $len = strlen($numeros);
    if ($len === 12 && strpos($numeros, '584') === 0) return $numeros;
    if ($len === 11 && strpos($numeros, '04') === 0) return '58' . substr($numeros, 1);
    if ($len === 10 && strpos($numeros, '4') === 0) return '58' . $numeros;
    return $numeros;
}

/**
 * Verifica si un teléfono venezolano ya existe en la BD.
 */
function telefono_existe($conn, $telefono) {
    $telefono_normalizado = normalizar_telefono_venezolano($telefono);
    if (empty($telefono_normalizado)) return true;

    $stmt = $conn->prepare("SELECT COUNT(*) FROM clientes WHERE 
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ? OR
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ? OR
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ?");
    
    $telefono_sin_prefijo = substr($telefono_normalizado, 2);
    $stmt->execute([
        "%" . $telefono_normalizado,
        "%" . $telefono_sin_prefijo,
        "%0" . $telefono_sin_prefijo
    ]);
    return $stmt->fetchColumn() > 0;
}

/**
 * (NUEVA FUNCIÓN) Parsea y formatea una fecha que puede venir de Excel.
 */
function formatear_fecha($fecha_excel) {
    if (empty($fecha_excel)) {
        return null;
    }
    // Si es numérico (fecha serial de Excel)
    if (is_numeric($fecha_excel) && $fecha_excel > 1) {
        // 25569 es la diferencia en días entre 1970-01-01 (epoch Unix) y 1900-01-01 (epoch Excel)
        $unix_timestamp = ($fecha_excel - 25569) * 86400;
        return gmdate('Y-m-d H:i:s', $unix_timestamp);
    }
    // Si es una cadena, intentar analizarla
    if (is_string($fecha_excel)) {
        try {
            $fecha_obj = new DateTime(trim($fecha_excel));
            return $fecha_obj->format('Y-m-d H:i:s');
        } catch (Exception $e) {
            return null; // Si el formato es irreconocible, devuelve null.
        }
    }
    return null;
}

/**
 * (FUNCIÓN MODIFICADA) Inserta un cliente en la base de datos, manejando la fecha de contacto.
 */
function insertar_cliente($conn, $cliente_data, $usuario_id) {
    $fecha_formateada = formatear_fecha($cliente_data['fecha'] ?? null);

    $nombre = limpiar_nombre($cliente_data['nombre'] ?? '');
    $telefono = normalizar_telefono_venezolano($cliente_data['telefono'] ?? '');
    $ciudad = limpiar_texto($cliente_data['ciudad'] ?? '');
    $origen = limpiar_texto($cliente_data['origen'] ?? '') ?: 'importacion';
    
    // Si tenemos una fecha válida del excel, la usamos.
    if ($fecha_formateada) {
        $sql = "INSERT INTO clientes (nombre, telefono, ciudad, origen, estado, estado_global, fecha_primer_contacto, creado_por_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $params = [$nombre, $telefono, $ciudad, $origen, 'nuevo', 'Importado desde Excel', $fecha_formateada, $usuario_id];
    } else {
        // Si no, dejamos que la BD use el valor por defecto (current_timestamp)
        // No incluimos 'fecha_primer_contacto' en la consulta
        $sql = "INSERT INTO clientes (nombre, telefono, ciudad, origen, estado, estado_global, creado_por_id) VALUES (?, ?, ?, ?, ?, ?, ?)";
        $params = [$nombre, $telefono, $ciudad, $origen, 'nuevo', 'Importado desde Excel', $usuario_id];
    }
    
    $stmt = $conn->prepare($sql);
    return $stmt->execute($params);
}

function limpiar_nombre($nombre) {
    if (empty(trim($nombre))) return 'Sin Nombre';
    $nombre = preg_replace('/[\x{1F300}-\x{1F5FF}\x{1F900}-\x{1F9FF}\x{1F600}-\x{1F64F}\x{1F680}-\x{1F6FF}\x{2600}-\x{26FF}\x{2700}-\x{27BF}\x{1F1E6}-\x{1F1FF}]/u', '', $nombre);
    $nombre = preg_replace('/[^\p{L}\p{N}\s\.\-]/u', '', $nombre);
    $nombre = preg_replace('/\s+/', ' ', trim($nombre));
    $nombre = mb_convert_case($nombre, MB_CASE_TITLE, 'UTF-8');
    return !empty($nombre) ? $nombre : 'Sin Nombre';
}

function limpiar_texto($texto) {
    if (empty($texto)) return '';
    $texto = preg_replace('/[\x{1F300}-\x{1F5FF}\x{1F900}-\x{1F9FF}\x{1F600}-\x{1F64F}\x{1F680}-\x{1F6FF}\x{2600}-\x{26FF}\x{2700}-\x{27BF}\x{1F1E6}-\x{1F1FF}]/u', '', $texto);
    return preg_replace('/\s+/', ' ', trim($texto));
}

switch ($action) {
    case 'importar_clientes_lote':
        try {
            $clientes = json_decode($_POST['clientes'] ?? '[]', true);
            if (!is_array($clientes)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }

            $stats = [
                'processed' => 0, 
                'success' => 0, 
                'duplicates' => 0, 
                'errors' => 0, 
                'omitted' => 0,
                'duplicatesList' => [], 
                'omittedList' => [],
                'errorsList' => []
            ];
            
            $conn->beginTransaction();

            foreach ($clientes as $index => $cliente_data) {
                $stats['processed']++;
                $telefono_original = $cliente_data['telefono'] ?? '';

                if (empty(trim($telefono_original))) {
                    $stats['errors']++;
                    $stats['errorsList'][] = [
                        'fila' => $cliente_data['fila_original'] ?? $index + 1,
                        'nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''),
                        'telefono' => $telefono_original,
                        'error' => 'Teléfono vacío'
                    ];
                    continue;
                }

                if (es_telefono_venezolano($telefono_original)) {
                    if (telefono_existe($conn, $telefono_original)) {
                        $stats['duplicates']++;
                        $stats['duplicatesList'][] = [
                            'nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''),
                            'telefono' => $telefono_original,
                            'fila' => $cliente_data['fila_original'] ?? $index + 1
                        ];
                    } else {
                        if (insertar_cliente($conn, $cliente_data, $usuario_id)) {
                            $stats['success']++;
                        } else {
                            $stats['errors']++;
                            $stats['errorsList'][] = [
                                'fila' => $cliente_data['fila_original'] ?? $index + 1,
                                'nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''),
                                'telefono' => $telefono_original,
                                'error' => 'Error al insertar en BD'
                            ];
                        }
                    }
                } else {
                    $stats['omitted']++;
                    $stats['omittedList'][] = [
                        'nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''),
                        'telefono' => $telefono_original,
                        'fila' => $cliente_data['fila_original'] ?? $index + 1
                    ];
                }
            }

            $conn->commit();
            
            if (function_exists('registrar_accion')) {
                registrar_accion('importar_clientes_excel', 
                    "Lote procesado: {$stats['success']} exitosos, {$stats['duplicates']} duplicados, {$stats['omitted']} omitidos, {$stats['errors']} errores de " . count($clientes) . " registros."
                );
            }

            $response = [
                'success' => true, 
                'message' => 'Lote procesado correctamente.', 
                'stats' => $stats,
                'batch_info' => [
                    'total_enviado' => count($clientes),
                    'total_procesado' => $stats['processed'],
                    'timestamp' => date('Y-m-d H:i:s')
                ]
            ];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al procesar lote: ' . $e->getMessage();
            error_log("Error en importar_clientes_lote: " . $e->getMessage() . " - Línea: " . $e->getLine());
        }
        break;

    case 'verificar_duplicados_previo':
        try {
            $telefonos = json_decode($_POST['telefonos'] ?? '[]', true);
            if (!is_array($telefonos)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }

            $duplicados = [];
            $venezolanos = [];
            $no_venezolanos = [];

            foreach ($telefonos as $telefono) {
                if (es_telefono_venezolano($telefono)) {
                    $venezolanos[] = $telefono;
                    if (telefono_existe($conn, $telefono)) {
                        $duplicados[] = normalizar_telefono_venezolano($telefono);
                    }
                } else {
                    $no_venezolanos[] = $telefono;
                }
            }

            $response = [
                'success' => true, 
                'duplicados' => array_unique($duplicados),
                'estadisticas' => [
                    'total_verificados' => count($telefonos),
                    'venezolanos' => count($venezolanos),
                    'no_venezolanos' => count($no_venezolanos),
                    'duplicados_encontrados' => count(array_unique($duplicados))
                ]
            ];

        } catch (Exception $e) {
            $response['message'] = 'Error al verificar duplicados: ' . $e->getMessage();
            error_log("Error en verificar_duplicados_previo: " . $e->getMessage());
        }
        break;

    case 'validar_estructura_datos':
        try {
            $datos = json_decode($_POST['datos_muestra'] ?? '[]', true);
            if (!is_array($datos)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }

            $estadisticas = [
                'total_filas' => count($datos),
                'filas_con_telefono' => 0,
                'telefonos_venezolanos' => 0,
                'telefonos_no_venezolanos' => 0,
                'filas_vacias' => 0,
                'campos_detectados' => []
            ];

            foreach ($datos as $fila) {
                if (empty(array_filter($fila))) {
                    $estadisticas['filas_vacias']++;
                    continue;
                }

                $telefono_encontrado = false;
                foreach ($fila as $celda) {
                    if (!empty($celda) && preg_match('/[0-9]/', $celda)) {
                        $telefono_limpio = preg_replace('/[^0-9]/', '', $celda);
                        if (strlen($telefono_limpio) >= 7) {
                            $telefono_encontrado = true;
                            if (es_telefono_venezolano($celda)) {
                                $estadisticas['telefonos_venezolanos']++;
                            } else {
                                $estadisticas['telefonos_no_venezolanos']++;
                            }
                            break;
                        }
                    }
                }

                if ($telefono_encontrado) {
                    $estadisticas['filas_con_telefono']++;
                }
            }

            $response = [
                'success' => true,
                'estadisticas' => $estadisticas,
                'recomendaciones' => [
                    'procesable' => $estadisticas['filas_con_telefono'] > 0,
                    'porcentaje_venezolanos' => $estadisticas['filas_con_telefono'] > 0 ? 
                        round(($estadisticas['telefonos_venezolanos'] / $estadisticas['filas_con_telefono']) * 100, 2) : 0
                ]
            ];

        } catch (Exception $e) {
            $response['message'] = 'Error al validar estructura: ' . $e->getMessage();
            error_log("Error en validar_estructura_datos: " . $e->getMessage());
        }
        break;

    case 'obtener_estadisticas_importacion':
        try {
            $limite_fecha = date('Y-m-d H:i:s', strtotime('-24 hours'));
            
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(*) as total_importados_hoy,
                    COUNT(DISTINCT creado_por_id) as usuarios_activos,
                    MAX(fecha_primer_contacto) as ultima_importacion
                FROM clientes 
                WHERE fecha_primer_contacto >= ? 
                AND estado_global LIKE '%Excel%'
            ");
            $stmt->execute([$limite_fecha]);
            $stats = $stmt->fetch(PDO::FETCH_ASSOC);

            $response = [
                'success' => true,
                'estadisticas_24h' => $stats
            ];

        } catch (Exception $e) {
            $response['message'] = 'Error al obtener estadísticas: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = 'Acción desconocida: ' . $action;
        break;
}

echo json_encode($response);